#include <assert.h>
#include <math.h>  
#include <stdio.h>

#include "constants.hh"
#include "model.hh"
#include "robot.hh"
#include "triangle.hh"

Triangle::Triangle(Coords c0p, Coords c1p, Coords c2p)
{
	push_back(c0p);
	push_back(c1p);
	push_back(c2p);
}

// There's a bunch of code stolen from update().  Must
// be a better way...
#define SWAPFLT(a, b) do {float tmp = a; a = b; b = tmp;} while (0)
float Triangle::match(Mtype matchtype, int color,
					  float belief_p, float disbelief_p,
					  Robot &robot)
{
	float matchval, tmatch;
	float tmp, x3, minx, maxx;
	float x, y;
	ModelPlane& plane = robot.get_model()[color];

	switch (matchtype) {
	  case MIN:
		matchval = 1.0;
		break;

	  case MAX:
		matchval = 0.0;
		break;

	  default:
		assert(false);
	}

	// life is a little easier if I extract and transform the
	// coordinates from the triangle first
	
	float ssin = sin(robot.get_angle());
	float scos = cos(robot.get_angle());
		   
	// work up the coordinates of the triangle in the model
	// here's another problem:  there are at least three places in the
	// code where I'm doing rotations and translations.  I really do
	// know better than that...  and I'm paying now.
	float x0 = (robot.get_x() +
				(*this)[0].get_x().eval(robot)*scos -
				(*this)[0].get_y().eval(robot)*ssin);
	float y0 = (robot.get_y() +
				(*this)[0].get_x().eval(robot)*ssin +
				(*this)[0].get_y().eval(robot)*scos);
	float x1 = (robot.get_x() +
				(*this)[1].get_x().eval(robot)*scos -
				(*this)[1].get_y().eval(robot)*ssin);
	float y1 = (robot.get_y() +
				(*this)[1].get_x().eval(robot)*ssin +
				(*this)[1].get_y().eval(robot)*scos);
	float x2 = (robot.get_x() +
				(*this)[2].get_x().eval(robot)*scos -
				(*this)[2].get_y().eval(robot)*ssin);
	float y2 = (robot.get_y() +
				(*this)[2].get_x().eval(robot)*ssin +
				(*this)[2].get_y().eval(robot)*scos);

	//printf("%f, (%f, %f) [%f, %f] <(%f %f) (%f %f) (%f %f)>\n",
	//	   robot.get_angle(),
	//	   robot.get_x(), robot.get_y(),
	//	   ssin, scos,
	//	   x0, y0, x1, y1, x2, y2);
	
	// Sort the three points so 0 is top-most, 1 is middle, 2 is
	// bottom-most
	if (y1 < y0) {
		// y1 is above y0.
		SWAPFLT(x0, x1);
		SWAPFLT(y0, y1);
	}
	if (y2 < y1) {
		// y2 is above y1.
		SWAPFLT(x1, x2);
		SWAPFLT(y1, y2);

		if (y1 < y0) {
			// original y0 may have been below y2
			SWAPFLT(x0, x1);
			SWAPFLT(y0, y1);
		}
	}
	
	// x3 is the new point we'll use to break the triangle in two.
	// Note the degenerate special case of y0 == y1 == y2...
	if (y0 == y2)
		x3 = x0;
	else
		x3 = x0 + (x2 - x0)*(y1-y0)/(y2-y0);

	// sort x1 and x3
	if (x1 > x3)
		SWAPFLT(x1, x3);

	// top half
	float ymin = minimum(maximum(0, y0), robot.get_arena().get_height()-1);
	float ymax = minimum(maximum(0, y1), robot.get_arena().get_height()-1);
	for (y = ymin; y < ymax; y++) {
		minx = minimum(maximum(0, x0 + (y-y0)*(x1-x0)/(y1-y0)),
					   robot.get_arena().get_width()-1);
		maxx = minimum(maximum(0, x0 + (y-y0)*(x3-x0)/(y1-y0)),
					   robot.get_arena().get_width()-1);

		for (x = minx; x < maxx; x++) {
			//if ((x == 14) && (y == 28))
			//	printf("this is the pixel!\n");
			//printf("belief (%f, %f), point (%d, %d), belief (%f, %f)\n",
			//	   belief_p, disbelief_p, (int) x, (int) y,
			//	   plane.get_point((int) x, (int) y).get_belief(),
			//	   plane.get_point((int) x, (int) y).get_disbelief());
			
			tmatch = plane.get_point(x, y).get_belief() * belief_p +
			  plane.get_point(x, y).get_disbelief() * disbelief_p;
			
			switch (matchtype) {
			  case MIN:
				matchval = minimum(matchval, tmatch);
				break;

			  case MAX:
				matchval = maximum(matchval, tmatch);
				break;

			  default:
				assert(false);
			}
		}
	}

	// bottom half
	ymin = ymax;
	ymax = minimum(maximum(0, y2), robot.get_arena().get_height()-1);
	for (y = ymin; y < ymax; y++) {
		minx = minimum(maximum(0, x1 + (y-y1)*(x2-x1)/(y2-y1)),
					   robot.get_arena().get_width()-1);
		maxx = minimum(maximum(0, x3 + (y-y1)*(x2-x3)/(y2-y1)),
					   robot.get_arena().get_width()-1);
		
		for (x = minx; x < maxx; x++) {
			//if ((x == 14) && (y == 28))
			//	printf("this is the pixel!\n");
			float b = plane.get_point(x, y).get_belief();
			float d = plane.get_point(x, y).get_disbelief();
			tmatch = b * belief_p + d * disbelief_p;
			
			switch (matchtype) {
			  case MIN:
				matchval = minimum(matchval, tmatch);
				break;

			  case MAX:
				matchval = maximum(matchval, tmatch);
				break;

			  default:
				assert(false);
			}
		}
	}

	return matchval;
}

void Triangle::update(Objtype objtype, int color,
					  float belief_p, float disbelief_p,
					  Robot &robot)
{
	float tmp, x3, minx, maxx;
	float x, y;
	float x0, y0, x1, y1, x2, y2;
	float mass, moment;
				
	//	printf("orig: <(%f %f) (%f %f) (%f %f)>\n",
	//		   (*this)[0].get_x().eval(robot),
	//		   (*this)[0].get_y().eval(robot),
	//		   (*this)[1].get_x().eval(robot),
	//		   (*this)[1].get_y().eval(robot),
	//		   (*this)[2].get_x().eval(robot),
	//		   (*this)[2].get_y().eval(robot));
	
	// The motors are special case colors; we only need to maintain
	// their moment and mass rather than their actual geometry.  Also,
	// the are robot-centric while the triangles for the "real" model
	// are model-centric.  So we're going to have some switch
	// statements here....

	// first, get the coordinates.  If this is a motor rule we stay
	// robot-centric; if it's a model rule we convert to model coords
	switch (objtype) {
		float ssin;
		float scos;
		
	  case MOTOR:
		// get triangle coordinates
		x0 = (*this)[0].get_x().eval(robot);
		y0 = (*this)[0].get_y().eval(robot);
		x1 = (*this)[1].get_x().eval(robot);
		y1 = (*this)[1].get_y().eval(robot);
		x2 = (*this)[2].get_x().eval(robot);
		y2 = (*this)[2].get_y().eval(robot);;

		break;

	  case OBJECT:
		ssin = sin(robot.get_angle());
		scos = cos(robot.get_angle());
		
		// work up the coordinates of the triangle in the model
		x0 = (robot.get_x() +
			  (*this)[0].get_x().eval(robot)*scos -
			  (*this)[0].get_y().eval(robot)*ssin);
		y0 = (robot.get_y() +
			  (*this)[0].get_x().eval(robot)*ssin +
			  (*this)[0].get_y().eval(robot)*scos);
		x1 = (robot.get_x() +
			  (*this)[1].get_x().eval(robot)*scos -
			  (*this)[1].get_y().eval(robot)*ssin);
		y1 = (robot.get_y() +
			  (*this)[1].get_x().eval(robot)*ssin +
			  (*this)[1].get_y().eval(robot)*scos);
		x2 = (robot.get_x() +
			  (*this)[2].get_x().eval(robot)*scos -
			  (*this)[2].get_y().eval(robot)*ssin);
		y2 = (robot.get_y() +
			  (*this)[2].get_x().eval(robot)*ssin +
			  (*this)[2].get_y().eval(robot)*scos);
		break;

	  default:
		assert(false);

	}
		
	//	printf("trans: %f, (%f, %f) [%f, %f] <(%f %f) (%f %f) (%f %f)>\n",
	//		   robot.get_angle(),
	//		   robot.get_x(), robot.get_y(),
	//		   ssin, scos,
	//		   x0, y0, x1, y1, x2, y2);
		
	// Sort the three points so 0 is top-most, 1 is middle, 2 is
	// bottom-most
	if (y1 < y0) {
		// y1 is above y0.
		SWAPFLT(x0, x1);
		SWAPFLT(y0, y1);
	}
	if (y2 < y1) {
		// y2 is above y1.
		SWAPFLT(x1, x2);
		SWAPFLT(y1, y2);
		
		if (y1 < y0) {
			// original y0 may have been below y2
			SWAPFLT(x0, x1);
			SWAPFLT(y0, y1);
		}
	}
	
	// x3 is the new point we'll use to break the triangle in two.
	// Note the degenerate special case of y0 == y1 == y2...
	if (y0 == y2)
		x3 = x0;
	else
		x3 = x0 + (x2 - x0)*(y1-y0)/(y2-y0);
	
	// sort x1 and x3
	if (x1 > x3)
		SWAPFLT(x1, x3);
	
	//	printf("(%f %f) [%f, %f] <(%f %f) (%f %f) (%f %f)>\n",
	//		   robot.get_x(), robot.get_y(), ssin, scos, x0, y0, x1, y1, x2, y2);

	ModelPlane& plane = robot.get_model()[color];
	switch (objtype) {
		float ymin, ymax;
		
	  case MOTOR:
		// let's try simplifying:  only consider belief and CG.  Throw away area
		mass = belief_p; //* (y2 - y0) * (x3 - x1) / 2.0;
		moment = mass * (y0 + y1 + y2) / 3.0;

		robot.get_motor(color).update(mass, moment);
		break;

	  case OBJECT:
		plane = robot.get_model()[color];
		
		// This is a really ugly hack to make drawing more efficient:  the
		// model will have to know how much of the planes have been
		// modified on this cycle, so we only modify as much of the
		// display as necessary on each cycle.
		
		// This information ought to be stored with the drawingarea, not
		// the model, but getting access to the drawingarea from here is
		// even uglier than this.  I guess I can mumble about "hints" and
		// try to make it sound like it was planned....
		minx = x0;
		maxx = x0;
		if (x1 < minx)
			minx = x1;
		if (x1 > maxx)
			maxx = x1;
		if (x2 < minx)
			minx = x2;
		if (x2 > maxx)
			maxx = x2;
		if (x3 < minx)
			minx = x3;
		if (x3 > maxx)
			maxx = x3;
		robot.get_model().update_bounds(minx, maxx, y0, y2);
	
		// bottom half
		ymin = minimum(maximum(0, y0), robot.get_arena().get_height()-1);
		ymax = minimum(maximum(0, y1), robot.get_arena().get_height()-1);
		for (y = ymin; y < ymax; y++) {
			minx = minimum(maximum(0, x0 + (y-y0)*(x1-x0)/(y1-y0)),
						   robot.get_arena().get_width()-1);
			maxx = minimum(maximum(0, x0 + (y-y0)*(x3-x0)/(y1-y0)),
						   robot.get_arena().get_width()-1);
			
			for (x = minx; x < maxx; x++)
				plane.get_point(x, y).update(belief_p, disbelief_p);
		}
		
		// top half
		ymin = ymax;
		ymax = minimum(maximum(0, y2), robot.get_arena().get_height()-1);

		for (y = ymin; y < ymax; y++) {
			minx = minimum(maximum(0, x1 + (y-y1)*(x2-x1)/(y2-y1)),
						   robot.get_arena().get_width()-1);
			maxx = minimum(maximum(0, x3 + (y-y1)*(x2-x3)/(y2-y1)),
						   robot.get_arena().get_width()-1);
			
			for (x = minx; x < maxx; x++)
				plane.get_point(x, y).update(belief_p, disbelief_p);
		}
		break;

	  default:
		assert(false);
	}
}

void Triangle::print(FILE *ofile)
{
	fprintf(ofile, "            [");
	for (iterator i = begin(); i < end(); i++)
		i->print(ofile);
	fprintf(ofile, " ]\n");
}

void Triangles::print(FILE *ofile)
{
	if (size() != 0) {
		fprintf(ofile,  "{");
		for (iterator i = begin(); i != end(); ++i)
			i->print(ofile);
		fprintf(ofile, "}\n");
	}
}

