#include <math.h>

#include <gtkmm/adjustment.h>

#include "arena.hh"
#include "model.hh"
#include "robot.hh"
#include "rule.hh"
#include "window1.hh"

extern window1* window1;

Robot::Robot(Arena &arenap,
			 MoveRules& moveRulesp,
			 ObjectRules& objRulesp,
			 Model& modelp,
			 float initxp=0.0,
			 float inityp=0.0,
			 float initanglep=0.0,
			 int numSonarsp=8,
			 float spreadp=0.15,
			 float retprob=1.0, float marsprob=0.0,
			 unsigned int precision=2,
			 float radiusp=8.0,
			 float topspeedp=1.0) :
  arena(arenap),
  initx(initxp), inity(inityp),
  real_x(initxp), real_y(inityp),
  initd(initanglep),
  radius(radiusp),
  moveRules(moveRulesp),
  objRules(objRulesp),
  model(modelp),
  visible(true)
{
	//	model = new Model(widthp, heightp);
	for (int i = 0; i < 2; ++i)
		motor.push_back(*new Motor(topspeedp));
	for (int i = 0; i < numSonarsp; ++i)
		sonar.push_back(*new Sonar(&arenap, 
								   (((float) i))/((float)numSonarsp)*2.0*M_PI,
								   radiusp,
								   spreadp, retprob, marsprob, precision));
}

void Robot::update_position()
{
	float avspeed, lspeed, rspeed;

	old_x = real_x;
	old_y = real_y;
	
	// we'll make speed 8 be one pixel per cycle
	lspeed = motor[0].get_speed()/8.0;
	rspeed = motor[1].get_speed()/8.0;

	// update speed display
	window1->frame2->vscale1->set_value(lspeed);
	window1->frame2->vscale2->set_value(rspeed);
	
	// amount we move is given by average speed of the two motors.
	avspeed = (lspeed + rspeed) / 2.0;
	
	// new direction is given by a bit of trig.
	// let A = angle between old direction and new direction (radians)
	// Rl = radius of circle defined by movement of left wheel
	// Rr = radius of circle defined by movement of right wheel
	// 2R = distance between wheels
	// Dl = distance travelled by left wheel
	// Dr = distance travelled by right wheel

	// Dl = Rl * A
	// Dr = Rr * A = (Rl + 2R) * A

	// Rl = Dl/A
	// Dr = (Rl + 2R) * A
	// Dr = (Dl/A + 2R)* A
	// Dr = Dl + 2RA
	// (Dr - Dl)/2R = A
	direction = direction + (rspeed - lspeed)/(2.0*radius);

	if (direction < 0.0)
		direction += 2 * M_PI;
	if (direction >  2 * M_PI)
		direction -= 2 * M_PI;

	// I move forward in local Y, so rotating pi/2 gives the right effect
	real_x += avspeed * cos(direction + M_PI_2);
	real_y += avspeed * sin(direction + M_PI_2);
	
	//printf("dir %f, cos %f, sin %f, loc (%f, %f)\n", direction,
	//		   cos(direction + M_PI_2), sin(direction + M_PI_2), real_x, real_y);
}

void Robot::update_sonar()
{
	for (int i = 0; i < sonar.size(); ++i)
		sonar[i].take_reading(real_x, real_y, radius, direction);

	//	for (int i = 0; i < sonar.size(); ++i)
	//		printf(" %3d", sonar[i].get_distance());
	//	printf("\n");
}

Arena& Robot::get_arena()
{
	return arena;
}

Model& Robot::get_model()
{
	return model;
}

int Robot::num_sonars()
{
	return sonar.size();
}

Sonar& Robot::get_sonar(int i)
{
	return sonar[i];
}

Motor& Robot::get_motor(int i)
{
	return motor[i];
}

void Robot::reset()
{
	real_x = initx;
	real_y = inity;
	direction = initd;

	model.reset();
	model.set_location(real_x, real_y, direction);
}

bool Robot::step()
{
	if (real_y < 0)
		return false;

	update_position();
	update_sonar();
	moveRules.process(*this);
	for (std::vector<Motor>::iterator i = motor.begin();
		 i != motor.end(); ++i)
		i->reset();
	objRules.process(*this);

	return true;
}
