#ifndef _BELIEF_HH_
#define _BELIEF_HH_

#include <assert.h>

#include "constants.hh"

class Belief {
  private:
	float belief;
	float disbelief;
	
  public:
	Belief(float beliefp=0, float disbeliefp=0) :
	  belief(beliefp), disbelief(disbeliefp)
	{}
	
	void set(float belief_p, float disbelief_p)
	{
		assert(belief_p >= 0);
		assert(disbelief_p >= 0);
		assert(belief_p + disbelief_p < 1.0 + FUZZ);
		
		// let's never allow perfect certainty (this is really a hack since
		// floating point arithmetic will let the functions hit 1 when
		// they should really only approach it).  I don't have to worry
		// about either belief or disbelief hitting 0, since the equations
		// don't get "stuck" if that happens.
		if (belief_p > 1.0-FUZZ) belief_p = 1.0-FUZZ;
		if (disbelief_p > 1.0-FUZZ) disbelief_p = 1.0-FUZZ;
		
		belief = belief_p;
		disbelief = disbelief_p;
	}

	void update(float belief, float disbelief)
	{
		float oldT = get_belief();
		float oldF = get_disbelief();
		float oldTF = 1.0 - (oldT + oldF);
		
		float newTF = 1.0 - (belief + disbelief);
		
		//	printf("updating (%d, %d) from (%f, %f) ", x, y, oldT, oldF);
		// generate new T, F
		float R1 = oldT  * belief;
		float R2 = oldT  * disbelief;
		float R3 = oldT  * newTF;
		float R4 = oldF  * belief;
		float R5 = oldF  * disbelief;
		float R6 = oldF  * newTF;
		float R7 = oldTF * belief;
		float R8 = oldTF * disbelief;
		float R9 = oldTF * newTF;
		
		float newT = R1 + R3 + R7;
		float newF = R5 + R6 + R8;
		newTF = R9;
		float incompat = R2 + R4;
		
		//	printf("before normalization:  T %f, F %f, TF %f, incompat %f\n",
		//			   oldT, oldF, oldTF, incompat);
		
		newT = newT / (1.0 - incompat);
		newF = newF / (1.0 - incompat);
		newTF = newTF / (1.0 - incompat);
		
		set(newT, newF);
		//	printf("to (%f, %f)\n", newT, newF);
	}
	
	inline void update(Belief belp)
	{
		update(belp.get_belief(), belp.get_disbelief());
	}
	
	float get_belief()
	{
		return belief;
	}
	
	float get_disbelief()
	{
		return disbelief;
	}

	void print(FILE *ofile)
	{
		fprintf(ofile, " ( %f, %f)", belief, disbelief);
	}
};

#endif
