;	$Id: eclipse.el,v 1.6.2.1 2002/05/13 18:30:14 ks15 Exp $   

;;; eclipse.el --- major mode for editing and running Eclipse under Emacs

;; Copyright (C) 1986, 87, 2001, 02 Free Software Foundation, Inc.

;; Author: Thorsten Winterer <t.winterer@icparc.ic.ac.uk>
;; based on the Eclipse mode from
;; Helmut Simonis <Helmut.Simonis@parc-technologies.com>
;; which was based on the prolog mode from
;; Masanobu UMEDA <umerin@mse.kyutech.ac.jp>
;; Keywords: languages

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This package provides a major mode for editing Eclipse.  It knows
;; about Eclipse syntax and comments, and can send regions to an inferior
;; Eclipse interpreter process.

;; The main difference to the previous Prolog mode is the extensive syntax
;; colouring done for Eclipse.
;; It supports in particular the comment/2 facility by highlighting
;; sections and pieces of the text

;; Note that the indentation is *not properly working* inside a string that
;; spans multiple screen lines. This is particularly a problem inside the
;; description texts of the comment/2 facility. Indentation of a comment/2 
;; predicate should therefore be done carefully!

;; the system also knows about C-style comments
;; and indents properly after a ';'

;; Function keys:

;; TAB indents the current line. If the line is already correctly indented,
;;   a TAB character will be inserted
;; M-TAB or M-C-\ indent a region
;;   ( M-TAB, since it's more mnemonic than M-C-\,
;;     and Lisp completions are not needed in Eclipse )
;; S-TAB inserts TAB character
;; C-l re-centres and re-fontifies
;; C-c-c comments out a region
;; C-c-r uncomments a region
;; C-c-i inverts the commenting of a region
;; C-c-C-f toggles the auto-line-break mode
;; C-c-C-j toggles the auto-indent mode

;; C-c-C-e starts an inferior Eclipse process
;; C-c-C-b compiles the buffer
;; C-c-C-v compiles a region
;; C-c-C-y compiles a region and switches to the Eclipse process
;; C-c-C-g passes a (region as) command to the Eclipse process
;; C-c-C-q stops the Eclipse process
;; C-c-C-k kills the Eclipse process
;; C-c-C-t starts the Eclipse-Tk-Tools

;; Variables:

;; eclipse-indent-width   Describes the number of space characters inserted
;;    when increasing the indentation. the default value is 4.
;; eclipse-tab-width   Describes the number of space characters inserted at
;;    the beginning of a line, if its indented. The default is 8.
;; eclipse-tab-mode   If non-nil, tabs are used for indentation, otherwise
;;    space characters only. The default is nil.
;; eclipse-autolinebreak-selected   If non-nil, auto-line-break is used.
;;    The default is t.
;; eclipse-autoindent-selected   If non-nil, auto-indent is used.
;;    The default is t.
;; eclipse-backtab   Describes the key- sequence for "backtab". This seems to
;;    depend on what Emacs and whar GUI you use. If [backtab] does not work,
;;    try [S-kp-tab] or [S-tab]. Or use whatever you fancy: with this key,
;;    additional tab characters (or equivalent space characters) are inserted
;; for XEmacs: the colours for the fontification are defined in variables
;;    eclipse-*-face-val

;; There are more customisable variables, but they are less likely to be
;; changed. Check the customisation options for group eclipse.

;; Running of Eclipse in an inferior mode has not been thoroughly tested,
;; I normally use the tkeclipse environment.

;; This version has been tested on GNU emacs 20.7.1 for Solaris 2.7,
;;    and on GNU emacs 21.1.1 for Linux and Solaris 2.7.; also to some
;;    extent on XEmacs 21.1.14 for Linux and Solaris 2.7.
;; Your mileage may vary.


;; New:
;; the indentation section has been largely rewritten
;; the mode is now compatible with XEmacs (21)

;;; NOTE: If there is a problem with entering commands in the inferior 
;;; ECLiPSe process window, disable the line
;;;               (define-key map "\r" 'eclipse-next-line)
;;; in the definition of function eclipse-mode-commands


;; To do:
;; clean up code for auto-fill
;; make indentation available for inferior process on GNU emacs 21 (??)
;; introduce levels of fontification
;;    (also to speed up loading when using emacs 20 or XEmacs)
;; improve integration with comint

;; To do (long term):
;; learn elisp ;-)
;; rewrite in proper elisp code
;; speed up the regular expressions.
;;   (if possible... and if still needed, since GNU emacs 21 starts a lot
;;    faster)
;; speed-up the indentation of regions
;;   (but this requires specialised functions...)


;;; Code:

;; what Emacs we're running?

(defvar eclipse-emacs-21 (equal (substring (version) 0 12) "GNU Emacs 21"))
(defvar eclipse-xemacs (equal (substring (version) 0 6) "XEmacs"))

;; Definitions...

(defvar eclipse-mode-syntax-table nil)
(defvar eclipse-mode-abbrev-table nil)
(defvar eclipse-mode-map nil)

(defgroup eclipse nil
  "Major mode for editing and running Eclipse under Emacs"
  :group 'languages)

(defcustom eclipse-version 5.4
  "ECLiPSe Version number"
  :type 'float
  :group 'eclipse)

(defcustom eclipse-path ""
  "*Path where ECLiPSe can be found."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-program-name "eclipse"
  "*Program name for invoking an inferior Eclipse with `run-eclipse'."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-tktools-name "tktools"
  "*Program name for invoking Tcl/Tk-based tools for Eclipse."
  :type 'string
  :group 'eclipse)

(defvar eclipse-program-call
  (concat eclipse-path eclipse-program-name))

(defvar eclipse-tktools-call
  (concat eclipse-path eclipse-tktools-name))

(defcustom eclipse-tktools-lib-name "remote_tools"
  "Eclipse library for invoking Tcl/Tk-based tools for Eclipse."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-54-tktools-call "attach_tools(Host/Port,block,writeln([Host,Port])).\n"
  "Eclipse command for invoking Tcl/Tk-based tools for Eclipse 5.4 and later.

The first parameter of attach_tools/3 returns the Host and Port, 
the second parameter is the timeout in seconds (or 'block' for no timeout),
the third parameter is a predicate called after establishing the connection."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-53-tktools-call "attach_tools.\n"
  "Eclipse command for invoking Tcl/Tk-based tools for Eclipse 5.3 and earlier."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-run-tktools-func 'eclipse-run-tktools
  "Elisp function to extract Host and Port values from output and start tktools.
Is added to 'comint-preoutput-filter-functions, and must remove itself from this list when the output line containing host and port is processed."
;  :type 'string
  :group 'eclipse)

(defcustom eclipse-tktools-lib-pred "lib"
  "How remote_tools.pl shall be loaded: as library (lib) or module (use_module)."
  :type 'string
  :group 'eclipse)

(defvar eclipse-tktools-lib-call
  (concat eclipse-tktools-lib-pred "(" eclipse-tktools-lib-name ").\n"))

(defcustom eclipse-reconsult-string "reconsult(user).\n"
  "*(Re)Consult mode (for C-Prolog and Quintus Prolog). "
  :type 'string
  :group 'eclipse)

(defcustom eclipse-consult-string "consult(user).\n"
  "*Consult mode."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-compile-string "compile(user).\n"
  "*Compile mode."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-eof-string "end_of_file.\n"
  "*String that represents end of file for eclipse.
   nil means send actual operating system end of file."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-halt-string "halt.\n"
  "*Command that stops the eclipse process."
  :type 'string
  :group 'eclipse)

(defcustom eclipse-indent-width 4
  "Standard indentation in Eclipse buffers."
  :type 'integer
  :group 'eclipse)

(defcustom eclipse-tab-width 8
  "Minimum indentation in Eclipse buffers."
  :type 'integer
  :group 'eclipse)

(defcustom eclipse-backtab [backtab]
  "Definition of \"backtab\" for insertion of additional tabs."
  :type 'array
  :group 'eclipse)

(defcustom eclipse-tab-mode nil
  "Indentation in Eclipse buffers with spaces or tabs?
Set this variable to nil to insert only space characters.
To change the behaviour during editing, use \\[eclipse-tab-mode-toggle]."
;  :type '...
  :group 'eclipse)

(defcustom eclipse-autolinebreak-selected t
  "Automatic line-break in Eclipse buffer."
;  :type '...
  :group 'eclipse)

(defcustom eclipse-autoindent-selected t
  "Automatic indentation in Eclipse buffer."
;  :type '...
  :group 'eclipse)

(if (not eclipse-xemacs)
    t
  ;; ain't the XEmacs colors ugly? let's set them to GNU Emacs like values!
  (defcustom eclipse-builtin-face-val "magenta"
    "Builtin face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-warning-face-val "red"
    "Warning face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-keyword-face-val "purple"
    "Keyword face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-function-name-face-val "blue"
    "Function name face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-variable-name-face-val "darkorange"
    "Variable name face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-comment-face-val "red3"
    "Comment face colour for XEmacs."
    :type 'string
    :group 'eclipse)
  ;; for whatever reason, XEmacs does not always use this value...
  (defcustom eclipse-string-face-val "salmon"
    "String face colour for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-constant-face-val "aquamarine4"
    "Constant face for XEmacs."
    :type 'string
    :group 'eclipse)
  (defcustom eclipse-type-face-val "forestgreen"
    "Type face for XEmacs."
    :type 'string
    :group 'eclipse)
  ;; create the two faces that XEmacs doesn't know
  (make-face 'font-lock-builtin-face)
  (make-face 'font-lock-constant-face)
  (set-face-property 'font-lock-builtin-face
		     'foreground eclipse-builtin-face-val)
  (set-face-property 'font-lock-warning-face
		     'foreground eclipse-warning-face-val)
  (set-face-property 'font-lock-keyword-face
		     'foreground eclipse-keyword-face-val)
  (set-face-property 'font-lock-function-name-face
		     'foreground eclipse-function-name-face-val)
  (set-face-property 'font-lock-variable-name-face
		     'foreground eclipse-variable-name-face-val)
  (set-face-property 'font-lock-comment-face
		     'foreground eclipse-comment-face-val)
  (set-face-property 'font-lock-string-face
		     'foreground eclipse-string-face-val)
  (set-face-property 'font-lock-constant-face
		     'foreground eclipse-constant-face-val)
  (set-face-property 'font-lock-type-face
		     'foreground eclipse-type-face-val))

(if eclipse-mode-syntax-table
    ()
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?_ "w" table)
    (modify-syntax-entry ?\\ "\\" table)
    (modify-syntax-entry ?/ ". 14b" table)
    (modify-syntax-entry ?* ". 23b" table)
    (modify-syntax-entry ?+ "." table)
    (modify-syntax-entry ?- "." table)
    (modify-syntax-entry ?= "." table)
    (modify-syntax-entry ?% "<" table)
    (modify-syntax-entry ?\n ">" table)
    (modify-syntax-entry ?< "." table)
    (modify-syntax-entry ?> "." table)
    (modify-syntax-entry ?\' "." table)
    (setq eclipse-mode-syntax-table table)))

(define-abbrev-table 'eclipse-mode-abbrev-table ())

(defvar eclipse-imenu-generic-expression
      '((nil 
	 "^\\([a-z][a-zA-Z0-9_]*\\)" 1))
  "Imenu generic expression for Eclipse mode. See `imenu-generic-expression'.")

(defconst eclipse-font-lock-keywords (purecopy
  (list
   '("\\([0-9]+\\)\\('\\)\\([a-zA-Z0-9]+\\)" 2 font-lock-builtin-face)
   ;; base operator

;;   '("^[^\"\n%]*\\(\"\\(\\\\\"\\|[^\"\n]\\)*\"\\)*[^\"\n%]*\\(%.*\\|/\\*\\([^*]\\)*\\*/\\)" . (3 font-lock-comment-face)) ;; comments
;;   '("\\('\\(\\\\'\\|[^']\\)*'\\|\"\\(\\\\\"\\|[^\"]\\)*\"\\)" . font-lock-string-face) ;; strings and quoted atoms

   '("/\\*[^*]*\\*/" 0 font-lock-comment-face t)
   ;; comments

   '("'\\(\\\\'\\|[^'\n]\\)*'" 0 font-lock-string-face)
   ;; quoted atoms

   '("\\<\\(a\\(bolish\\|rr_create\\|ssert[az]?\\|ttach_suspensions\\)\\|c\\(all_priority\\|urrent_array\\)\\|de\\(cval\\|fine_macro\\)\\|e\\(r\\(ase\\(_\\(a\\(ll\\|rray\\)\\|macro\\)\\)?\\|ror\\)\\|vent\\(_after\\(_every\\)?\\)?\\)\\|getval\\|in\\(c\\(lude\\|val\\)\\|it_suspension_list\\|sert_suspension\\)\\|kill_suspension\\|local_array\\|m\\(ake_suspension\\|e\\(rge_suspension_list\\|ta_attribute\\)\\)\\|notify_constrained\\|re\\(cord[az]?\\|record\\|set_event_handler\\|tract\\(_all\\|all\\)?\\)\\|s\\(chedule_suspensions\\|et\\(_\\(chtab\\|default_error_handler\\|e\\(rror_handler\\|vent_handler\\)\\|flag\\|interrupt_handler\\|suspension_data\\|var_bounds\\)\\|arg\\(\\)?\\|val\\)\\|uspend\\)\\|t\\(est_and_setval\\|rigger\\)\\|update_struct\\|x\\(get\\|set\\)\\)(" 1 font-lock-warning-face)
   ;; critical builtins
   '("\\(\\<\\(abort\\|fa\\(il\\|lse\\)\\|halt\\|make\\|not\\|pause\\|re\\(peat\\|set_error_handlers\\)\\|wake\\)\\>\\|[ \t(]\\(\\\\\\+\\|~\\)[ \t(]\\|!\\|-->\\|{\\|}\\|\\\\\\+\\|-\\?->\\|\\?-\\)" 0 font-lock-warning-face)
   ;; critical builtins with arity 0 or prefix op
   ;; special for cut, not   (\\?-\\s-*$)???
   '("^\\(:-\\s-*\\(mode\\|tool\\)\\)\\(.*\\)$" 3 font-lock-warning-face)
   ;; highlight mode/tool declaration

   '("\\([ \t]^[ \t]\\|\\<once\\(\\>\\|(\\)\\|\\<\\(env\\|garbage_collect\\|listing\\|nl\\|statistics\\|trimcore\\)\\>\\)" 0 font-lock-builtin-face)
   '("\\<\\('C'\\|a\\(bs\\|c\\(cept\\|os\\)\\|dd_attribute\\|l\\(arm\\|s\\)\\|ppend\\(_strings\\)?\\|rg[cv]?\\|sin\\|t\\(_eof\\|an\\|om\\(_\\(length\\|string\\)\\|ic\\)?\\)?\\)\\|b\\(ag\\(_\\(create\\|dissolve\\|enter\\)\\|of\\)\\|etween\\|ind\\|lock\\|real\\(_\\(bounds\\|from_bounds\\|m\\(ax\\|in\\)\\)\\)?\\|ytes_to_term\\)\\|c\\(a\\(ll\\(_c\\)?\\|n\\(cel_after_event\\|onical_path_name\\)\\)\\|d\\|eiling\\|har_\\(code\\|int\\)\\|l\\(ause\\|ose\\|rbit\\)\\|o\\(mp\\(are\\(_instances\\)?\\|ile\\(_\\(stream\\|term\\)\\|d_stream\\)?\\|ound\\)\\|n\\(cat_\\(atoms?\\|strings?\\)\\|nect\\)\\|py_term\\(_vars\\)?\\|s\\|unt\\|verof\\)\\|putime\\|reate_module\\|urrent_\\(a\\(fter_event\\|tom\\)\\|built_in\\|compiled_file\\|error\\|functor\\|host\\|interrupt\\|m\\(acro\\|odule\\(_predicate\\)?\\)\\|op\\|predicate\\|record\\|s\\(tr\\(eam\\|uct\\)\\|uspension\\)\\)\\)\\|d\\(ate\\|e\\(l\\(ayed_goals\\(_number\\)?\\|ete\\)\\|mon\\|nominator\\)\\|i\\(m\\|splay\\)\\|om\\)\\|e\\(r\\(ase_module\\|r\\(no_id\\|or_id\\)\\)\\|val\\|x\\(ec\\(_group\\)?\\|i\\(st\\(ing_file\\|s\\)\\|t\\(_block\\)?\\)\\|p\\(and_goal\\)?\\|ternal\\)\\)\\|f\\(i\\(ndall\\|x\\)\\|l\\(o\\(at\\(\\)?\\|or\\)\\|ush\\)\\|or\\(each\\(arg\\)?\\|k\\)?\\|r\\(andom\\|ee\\|omto\\)\\|unctor\\)\\|get\\(0\\|_\\(ch\\(ar\\|tab\\)\\|e\\(rror_handler\\|vent_handler\\)\\|f\\(ile_info\\|lag\\)\\|interrupt_handler\\|module_info\\|priority\\|s\\(tream\\(_info\\)?\\|uspension_data\\)\\|var_\\(bounds\\|info\\)\\)\\|bit\\|cwd\\|env\\)?\\|help\\|i\\(n\\(line\\|stance\\|teger\\(_atom\\)?\\)\\|s_\\(built_in\\|dynamic\\|handle\\|predicate\\|record\\|suspension\\)\\)\\|join_string\\|k\\(eysort\\|ill\\)\\|l\\(ength\\|isten\\|n\\|o\\(ad\\|ck\\)\\)\\|m\\(ax\\|e\\(mber\\(chk\\)?\\|rge\\|ta\\(_\\(attribute\\|bind\\)\\)?\\)\\|in\\|kdir\\|od\\|sort\\|utex\\(_init\\)?\\)\\|n\\(ame\\|ew_socket_server\\|l\\|o\\(n\\(ground\\|var\\)\\|t_unify\\)\\|um\\(ber\\(_string\\)?\\|erator\\)\\)\\|o\\(ccurs\\|pen\\|s_file_name\\)\\|p\\(a\\(ram\\|thname\\)\\|eer\\(_\\(get_property\\|queue_\\(c\\(lose\\|reate\\)\\|get_property\\)\\)\\)?\\|hrase\\|ipe\\|lus\\|ortray_goal\\|r\\(ed\\|intf?\\|ofile\\|une_instances\\)\\|ut\\(_char\\)?\\)\\|r\\(a\\(ndom\\|tional\\)\\|e\\(a\\(d\\(_\\(directory\\|exdr\\|string\\|token\\)\\|var\\)\\|[dl]\\)\\|corded\\(_list\\)?\\|ferenced_record\\|mote_\\(connect\\(_\\(accept\\|setup\\)\\)?\\|disconnect\\|yield\\)\\|name\\|verse\\)\\|ound\\)\\|s\\(e\\(e[dk]\\|lect\\|t\\(_stream\\(_property\\)?\\|bit\\|of\\)\\)\\|h\\(elf_\\(abolish\\|create\\|get\\|set\\)\\)?\\|in\\|leep\\|o\\(cket\\|rt\\)\\|plit_string\\|qrt\\|t\\(atistics\\|ring\\(_l\\(ength\\|ist\\)\\)?\\)\\|u\\(b\\(call\\|s\\(cript\\|tring\\)\\)\\|m\\|spensions\\)\\|ystem\\)\\|t\\(an\\|erm_\\(hash\\|string\\|to_bytes\\|variables\\)\\|imes\\|ool_body\\|y\\(pe_of\\|[io]\\)\\)\\|un\\(get\\|lock\\)\\|var\\(iant\\)?\\|w\\(ait\\|rite\\(_\\(canonical\\|exdr\\)\\|clause\\|ln\\|q\\)?\\)\\|xor\\|yield\\)(" 1 font-lock-builtin-face)
   ;; normal built-ins & control : non-critical
   '("\\(\\(\\<do\\|->\\|:-\\)\\s-*$\\|\\<is\\>\\|->\\|=\\.\\.\\|==\\|=\\\\=\\|=:=\\|\\\\==\\|[#@$`]?\\(>=\\|>\\|=<\\|<=\\|<\\|=\\|\\\\=\\)\\|#\\(#\\|\\\\/\\|/\\\\\\)\\|::\\|`::\\|`<>\\|~=\\|[ \t(]\\([-*+/^]\\|//\\|/\\\\\\|<<\\|>>\\|\\\\/\\)(\\|;\\)" 0 font-lock-builtin-face)
   ;; operators at end of line & infix built-ins & arithmetic built-ins &
   ;; semicolon

   '("^:-\\s-*\\(begin_module\\|comment\\|dynamic\\|e\\(nsure_loaded\\|xport\\)\\|import\\|l\\(ib\\|ocal \\(macro\\|op\\|portray\\|reference\\|struct\\|variable\\)\\)\\|mod\\(e\\|ule\\(_interface\\)?\\)\\|p\\(arallel\\|ragma\\)\\|reexport\\|tool\\|use_module\\)\\>" 0 font-lock-keyword-face)
   ;; directives

   '("\\<\\(debug\\(ging\\)?\\|no\\(debug\\|spy\\|trace\\)\\|s\\(kipped\\|py\\)\\|trace\\(able\\)?\\|un\\(skipped\\|traceable\\)\\)\\>" 0 font-lock-constant-face)
   ;; debugging
   '("\\<\\(debug\\|[gs]et_leash\\|\\(kill\\|make\\)_display_matrix\\|trace\\(_port\\)?\\)(" 1 font-lock-constant-face)
   ;; debugging
   '("^:-\\s-*\\(module\\|module_interface\\|begin_module\\|use_module\\|lib\\|ensure_loaded\\)(\\([a-zA-Z_0-9]+\\))" 2 font-lock-constant-face)
   '("^:-\\s-*local \\(struct\\|macro\\|op\\|portray\\|reference\\)(\\([a-zA-Z_0-9,() ]+\\))" 2 font-lock-constant-face)
   ;; module names & structures

   ;; special treatment first, so the general cases are delayed until now
   '("\\<\\(op\\)(" 1 font-lock-warning-face)
   '(":-" 0 font-lock-builtin-face)
   ;; :- not necessarily at end of line

   '("\\(from\\)\\s-+\\([a-z0-9_]+\\)" 1 font-lock-keyword-face)
   ;; import from module
   '("\\(from\\)\\s-+\\([a-z0-9_]+\\)" 2 font-lock-constant-face)
   ;; import from module

   '("\\<[a-z][a-zA-Z0-9_]* \\(of [a-z][a-zA-Z0-9_]*\\|with\\)\\>" 0 font-lock-constant-face)
   ;; special case for structures
   '("\\([ \t]@[ \t]+[A-Za-z0-9_]+[ \t]*\\)[),;\n]" 1 font-lock-constant-face)
   ;;
   '("\\<\\(anti_first_fail\\|bbs\\|c\\(omplete\\|redit\\)\\|dbs\\|end_of_file\\|first_fail\\|in\\(domain\\(_\\(interval\\|m\\(ax\\|edian\\|i\\(ddle\\|n\\)\\)\\|random\\|split\\)\\)?\\|put_order\\)\\|l\\(argest\\|ds\\)\\|m\\(ax_regret\\|ost_constrained\\)\\|occurrence\\|smallest\\|true\\)\\>" 0 font-lock-constant-face)
   ;; special case for true and fail and other constant symbols

   '("\\<\\([_A-Z][a-zA-Z0-9_]*\\)\\>" 1 font-lock-variable-name-face)
   ;; show variables

   ;; fontification of comment predicate
   '("\\(^\\|comment(\\)\\s-*\\(a\\(lias\\|mode\\|rgs\\|uthor\\)\\|copyright\\|d\\(ate\\|esc\\)\\|e\\(g\\|xceptions\\)\\|fail_if\\|in\\(clude\\|dex\\)\\|resat\\|s\\(ee_also\\|ummary\\)\\|template\\)\\s-*[,:]" 2 font-lock-type-face)
   ;; special case for comment/2 predicate
   '("^:-\\s-*comment(\\([a-z_][a-z0-9_]*\\s-*/\\s-*[0-9]+\\)" 1 font-lock-function-name-face)
   ;; predicate definitions in comment/2
   '("^[a-z][a-zA-Z0-9_]*" 0 font-lock-function-name-face)
   ;; predicate definitions always start at bol
   '("\\(<[Pp]>\\|</[Pp]>\\|<[Bb]>\\|</[Bb]>\\|<[Ll][Ii]>\\|</[Ll][Ii]>\\|<[Uu][Ll]>\\|</[Uu][Ll]>\\|<[Aa][^>]*>\\|</[Aa]>\\|<[Ii]>\\|</[Ii]>\\|<[Dd][Ll]>\\|</[Dd][Ll]>\\|<[Dd][Tt]>\\|</[Dd][Tt]>\\|<[Dd][Dd]>\\|</[Dd][Dd]>\\|<[Tt][Tt]>\\|</[Tt][Tt]>\\|<[Ee][Mm]>\\|</[Ee][Mm]>\\|<[Pp][Rr][Ee]>\\|</[Pp][Rr][Ee]>\\)" 0 font-lock-function-name-face t)
   ;; override html markup in strings and comments
   '("\\(^\\s-*\\|[[]\\)\"\\([_A-Z][a-zA-Z0-9_]*\\)\"\\s-*:" 2 font-lock-variable-name-face t)
   ;; show variables in args field of comment, overrides comments
   ))
  "Additional expressions to highlight in Eclipse mode.")

(if eclipse-xemacs
    (put 'eclipse-mode 'font-lock-keywords '(eclipse-font-lock-keywords nil))
  (put 'eclipse-mode 'font-lock-defaults '(eclipse-font-lock-keywords nil)))
;; case sensitive

(defun eclipse-mode-variables ()
  (set-syntax-table eclipse-mode-syntax-table)
  (setq local-abbrev-table eclipse-mode-abbrev-table)
  (make-local-variable 'paragraph-start)
  (setq paragraph-start (concat "%%\\|$\\|" page-delimiter)) ;'%%..'
  (make-local-variable 'paragraph-separate)
  (setq paragraph-separate paragraph-start)
  (make-local-variable 'paragraph-ignore-fill-prefix)
  (setq paragraph-ignore-fill-prefix t)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'eclipse-indent-line)
  (make-local-variable 'comment-start)
  (setq comment-start "%")
  (make-local-variable 'comment-column)
  (setq comment-column 0)
  (make-local-variable 'imenu-case-fold-search)
  (setq imenu-case-fold-search t)
  (make-local-variable 'imenu-generic-expression)
  (setq imenu-generic-expression eclipse-imenu-generic-expression)
  (make-local-variable 'imenu-syntax-alist)
  (setq imenu-syntax-alist '(("+-*/.<>=?!$%_&~^:" . "w")))
  (make-local-variable 'comment-indent-function)
  (setq comment-indent-function 'eclipse-indent-line)
  t)

(defun eclipse-mode-commands (map)
  "Contains the key-bindings for the major ECLiPSe mode.
The following commands are available:

\\{eclipse-mode-map}"
  (define-key map "\r" 'eclipse-next-line)
  (define-key map "\t" 'eclipse-indent-line)
  (define-key map "\M-\t" 'eclipse-indent-region)
  (define-key map "\M-C-\\" 'eclipse-indent-region)
; <backtab> seems to be defined differently in almost every window
; manager. so you've got to customize it...
  (define-key map eclipse-backtab 'eclipse-insert-tab)
  (define-key map "\C-l" 'eclipse-recenter)
  (define-key map "\C-c\C-e" 'run-eclipse)
  (define-key map "\C-c\C-q" 'stop-eclipse)
  (define-key map "\C-c\C-k" 'kill-eclipse)
  (define-key map "\C-c\C-t" 'eclipse-start-tools)
  (define-key map "\C-c\C-g" 'eclipse-run-region)
  (define-key map "\C-c\C-b" 'eclipse-compile-buffer)
  (define-key map "\C-c\C-v" 'eclipse-compile-region)
  (define-key map "\C-c\C-y" 'eclipse-compile-region-and-go)
  (define-key map "\C-cc" 'eclipse-comment-region)
  (define-key map "\C-cr" 'eclipse-uncomment-region)
  (define-key map "\C-ci" 'eclipse-invert-comment-region)
  (define-key map "\C-c\C-f" 'eclipse-autolinebreak-toggle)
  (define-key map "\C-c\C-j" 'eclipse-autoindent-toggle)
  t)

(if eclipse-mode-map
    nil
  (setq eclipse-mode-map (make-sparse-keymap))
  (eclipse-mode-commands eclipse-mode-map))

;; define menus
(easy-menu-define
 eclipse-process-menu eclipse-mode-map
 "ECLiPSe-Process Menu in ECLiPSe mode.
Contains commands that are associated with an inferior ECLiPSe process."
 '("ECLiPSe-Process"
   ["Run ECLiPSe" run-eclipse t]
   ["Stop ECLiPSe" stop-eclipse t]
   ["Kill ECLiPSe" kill-eclipse t]
   "--"
   ["Compile buffer" eclipse-compile-buffer t]
   ["Compile region" eclipse-compile-region t]
   ["Run region" eclipse-run-region t]
   "--"
   ["Start TkTools" eclipse-start-tools t]) )

(easy-menu-define
 eclipse-edit-menu eclipse-mode-map
 "ECLiPSe-Edit Menu in ECLiPSe mode.
Contains commands that are associated with editing an ECLiPSe file."
 '("ECLiPSe-Edit"
   ["Indent line" eclipse-indent-line t]
   ["Indent region" eclipse-indent-region t]
   "--"
   ["Re-fontify & re-center" eclipse-recenter t]
   "--"
   ["Comment out region" eclipse-comment-region t]
   ["Uncomment region" eclipse-uncomment-region t]
   ["Invert commenting of region" eclipse-invert-comment-region t]
   "--"
   ("Toggle auto-line-break on/off"
    ["Auto-line-break" eclipse-autolinebreak-toggle
     :style toggle
     :selected eclipse-autolinebreak-selected])
   ("Toggle auto-indent on/off"
    ["Auto-indent" eclipse-autoindent-toggle
     :style toggle
     :selected eclipse-autoindent-selected])) )

;;;###autoload
(defun eclipse-mode ()
  "Major mode for editing Eclipse code for Eclipses.
Blank lines and `%%...' separate paragraphs.  `%'s start comments.

Commands:
\\{eclipse-mode-map}

Entry to this mode calls the value of `eclipse-mode-hook' if that value is non-nil.

The auto-line-break mode is set to on on start-up. It can be toggled by calling \\[eclipse-autolinebreak-toggle], or customised by setting the variable `eclipse-autolinebreak-selected'.
A non-nil value means that auto-line-break is on.

The auto-indent mode is set to on on start-up. It can be toggled by calling \\[eclipse-autoindent-toggle], or customised by setting the variable `eclipse-autoindent-selected'.
A non-nil value means that auto-indent is on.

The tab mode is set to \"use space characters\" on start-up. It can be toggled by calling \\[eclipse-tab-mode-toggle], or customised by setting the variable `eclipse-tab-mode'.
A non-nil value means that tab characters are used is possible, otherwise space characters only.

The width of the initial indentation at the beginning of a line is stored in the variable `eclipse-tab-width'. Further indentations, after open brackets, use the value stored in `eclipse-indent-width'.
Both variables are customisable.

The key for inserting additional tab characters (or the equivalent number of space characters) is defined in `eclipse-backtab'."
  (interactive)
  (kill-all-local-variables)
  (use-local-map eclipse-mode-map)
  (setq major-mode 'eclipse-mode)
  (setq mode-name "Eclipse")
  (eclipse-mode-variables)
  ;; Font lock support
  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(eclipse-font-lock-keywords nil))
  ;; (setq font-lock-keywords-only t)
  ;;    it's far too slow to fontify comments and strings by oneself
  ;;    may lead to stack overflow, too
  (setq indent-tabs-mode eclipse-tab-mode)
  (easy-menu-add eclipse-edit-menu)
  (easy-menu-add eclipse-process-menu)
  ;; always start with auto-line-break mode
  (if (not eclipse-autolinebreak-selected)
      t
    (auto-fill-mode 1)
    (setq auto-fill-function 'eclipse-auto-fill-function))
  (run-hooks 'eclipse-mode-hook))

;;
;; Eclipse mode auto-fill
;;

(defun eclipse-autolinebreak-toggle ()
  "Toggle auto-line-break on/off in ECLiPSe mode.

When auto-line-break is on, strings, comments, and expressions are broken up if they get too long to fit into a line."
  (interactive)
  (if (not auto-fill-function)
      (prog1 (setq auto-fill-function 'eclipse-auto-fill-function)
	(force-mode-line-update)
	(setq eclipse-autolinebreak-selected t))
    (setq auto-fill-function nil)
    (setq eclipse-autolinebreak-selected nil))
  t)

(defun eclipse-autoindent-toggle ()
  "Toggle auto-indent on/off in ECLiPSe mode.

When auto-indent is on, lines are automatically indented after pressing <RET>."
  (interactive)
  (if (not eclipse-autoindent-selected)
      (setq eclipse-autoindent-selected t)
    (setq eclipse-autoindent-selected nil))
  t)

(defun eclipse-next-line ()
  "This function is called when <RET> is pressed.
If auto-indent is on, the next lines is automatically indented."
  ;; handle process marks myself, since older comint.el do not have
  ;; functions-comint-set process-mark and comint-goto-process-mark
  (interactive)
  (let (string input-string (flag nil) proc
	beg aux end (name (buffer-name)))
    (cond ( (string-equal name "*eclipse*")
	    ;; in eclipse process buffer?
	    (cond
	     ;; normal eclipse process command?
	     ( (save-excursion
		 (if (eobp) (backward-char))
		 (while (looking-at "[ \t\n]")
		   (backward-char))
		 (setq beg (point))
		 (backward-char)
		 (looking-at "[^.]\\."))
	       (goto-char (+ 1 beg))
	       (setq end (point))
	       (goto-char
		(process-mark (get-buffer-process (current-buffer))))
	       (setq beg (point))
	       (eclipse-end-of-clause)
	       (setq string (buffer-substring beg (point)))
	       (while (> end (point))
		 (forward-line)
		 (beginning-of-line)
		 (skip-chars-forward " \t\n")
		 (setq aux (point))
		 (eclipse-end-of-clause)
		 (setq string (concat string (buffer-substring aux (point)))))
	       (insert "\n")
	       (setq string (concat string "\n"))
	       (eclipse-set-process-mark)
	       (if eclipse-emacs-21
		   (eclipse-change-face beg))
	       (process-send-string "eclipse" string)
	       (if eclipse-emacs-21
		   (comint-add-to-input-history string)
		 (ring-insert comint-input-ring string)) )
	     ;; eclipse debugger command?
	     ( (save-excursion
		 (if (not eclipse-emacs-21)
		     t
		   (beginning-of-line)
		   (backward-char))
		 (beginning-of-line)
		 (looking-at ".*%>"))
	       (save-excursion
		 (while (not (looking-at "%>"))
		   (backward-char))
		 (forward-char 3)
		 (setq beg (point)))
	       (setq string (buffer-substring beg (point)))
	       (if eclipse-emacs-21
		   (eclipse-change-face beg))
	       (insert "\n")
	       (eclipse-set-process-mark)
	       (process-send-string "eclipse" string) )
	     ;; eclipse interruption command?
	     ( (save-excursion
		 (beginning-of-line)
		 (if eclipse-emacs-21
		     (backward-char))
		 (beginning-of-line)
		 (looking-at "interruption"))
	       (save-excursion
		 (while (not (looking-at "\\?"))
		   (backward-char))
		 (forward-char 2)
		 (setq beg (point)))
	       (setq string (buffer-substring beg (point)))
	       (if eclipse-emacs-21
		   (eclipse-change-face beg))
	       (insert "\n")
	       (eclipse-set-process-mark)
	       (process-send-string "eclipse" string) )
	     ;; eclipse tracer command?
	     ( (save-excursion
		 (if (not eclipse-emacs-21)
		     t
		   (beginning-of-line)
		   (backward-char))
		 (beginning-of-line)
		 (looking-at ".*[]:?]")
		 (if (looking-at ".*\\[")
		     t
		   (if (looking-at "\\(set\\|.*[?]\\)")
		       (setq flag t)))
		 (setq proc (get-buffer-process (current-buffer)))
		 (setq beg (process-mark proc)))
	       (setq string (buffer-substring beg (point)))
	       (if eclipse-emacs-21
		   (eclipse-change-face beg))
	       (insert "\n")
	       (if flag
		   (setq string (concat string "\n")))
	       (eclipse-set-process-mark)
	       (process-send-string "eclipse" string) )
	     ;; else: regular line
	     ( t
	       (insert "\n")
	       (if eclipse-emacs-21
		   (insert "\t")  ;; if emacs 21, do not attempt to indent
		 (eclipse-indent-line)) )))
	  ;; else in eclipse code buffer
	  ( t
	    (newline)
	    (if eclipse-autoindent-selected
		(eclipse-indent-line t)) )))
  t)

(defun eclipse-set-process-mark ()
  ;; adapted from comint.el
  (let ((proc (get-buffer-process (current-buffer))))
    (set-marker (process-mark proc) (point))))

(defun eclipse-change-face (beg)
  ;; copied from comint.el: change face
  (let ((over (make-overlay beg (point) nil nil t)))
    (overlay-put over 'field 'input)
    (overlay-put over 'face 'comint-highlight-input)))

(defun eclipse-auto-fill-function ()
  "Auto-fill function for ECLiPSe mode.

Example:
pred(Var) :-
        another_pred(
                        \"This is a very long string that will be\"
                        \" automatically wrapped around\", %% This is a
                                                         %% comment
                        A + Really + Very + Long + And + Nonsensical +
                        Expression
                    )."
  ;; copied & adapted from simple.el: the breaking-up of strings, quoted
  ;;    atoms, and comments needs special handling
  ;; comments partly deleted: look up in simple.el
  (let (fc give-up (ep nil))
    (if (or (null (setq fc (current-fill-column)))
	    (<= (current-column) fc))
	nil ;; Auto-filling not required
      (while (and (not give-up) (> (current-column) fc))
	;; Determine where to split the line.
	(let* (bol-point first-break-point atom-break-point
	       (fill-point
		(let ((opoint (point)))
		  (save-excursion
		    (beginning-of-line)
		    (setq bol-point (point))
		    (move-to-column (1+ fc))
		    ;; Move back to the point where we can break the line.
		    (re-search-backward "[ \t,;+*-/=<>]")
		    ;; check if we're inside a quoted atom
		    ;; if so, break before the start of the quoted atom
		    (setq first-break-point (point))
		    (beginning-of-line)
		    (let ((cc nil) (ac nil) (sc nil))
			(while (< (point) first-break-point)
			  (forward-char 1)
			  (cond ( (looking-at "\"")
				  (if (or ac cc)
				      t
				    (if sc
					(setq sc nil)
				      (setq sc t))) )
				( (looking-at "%")
				  (if (or sc ac)
				      t
				    (setq cc t)) )
				( (looking-at "'")
				  (if (or sc cc)
				      t
				    (if ac
					(setq ac nil)
				      (setq ac t)
				      (save-excursion
					(re-search-backward "[ \t,;+*-/=<>]")
					(setq atom-break-point (point))))) )
				( t t )))
			(if ac
			    (goto-char atom-break-point)))
		    ;; If we find nowhere on the line to break it,
		    ;; break after one word.
		    (if (bolp)
			(progn
			  (re-search-forward "[ \t]" opoint t))
		      (if (looking-at "[ \t]")
			  ;; Break the line at word boundary.
			  (skip-chars-backward " \t")
			;; Break the line after/before \c|.
			(forward-char)))
		    (if (and enable-multibyte-characters
			     (not eclipse-xemacs))
			;; special function for the charset of non-ascii
			;; character to find the correct break point.
			;; Don't do in XEmacs, charset-after not available.
			(if (not (and (eq (charset-after (1- (point))) 'ascii)
				      (eq (charset-after (point)) 'ascii)))
			    (fill-find-break-point bol-point)))
		    ;; move back before any whitespace here.
		    (skip-chars-backward " \t")
		    ;; that's the fill-point
		    (point)))))
	  ;; See whether the place we found is any good.
	  (if (save-excursion
		(goto-char fill-point)
		(and (not (bolp))
		     ;; There is no use breaking at end of line...
		     (not (save-excursion (skip-chars-forward " \t") (eolp)))
		     ;; ...or beginning of line.
		     ;; (test for comment start deleted)
		     (not (bolp))))
	      ;; Ok, we have a useful place to break the line.  Do it.
	      (let (counter (colmn nil) (prev-column (current-column)))
		;; now we must determine, if the break-point is
		;; (a) in a comment, or
		;; (b) in a string, or
		;; (c) in a regular line
		;; if (a), break the line, insert spaces until the beginning
		;;         of the comment, and insert as many percentage signs
		;; if (b), add \", break the line, indent, add \"
		;; if (c), break the line and indent
		;; quoted atoms have been dealt with while finding the
		;; break point. dealing with comments should be done at that
		;; point, too...
		(cond ( (save-excursion
			  ;; inside a string?
			  (goto-char fill-point)
			  (setq counter 0)
			  (setq colmn nil)
			  (while (not (bolp))
			    (cond ( (looking-at "\"")
				    (setq counter (+ counter 1)) )
				  ( (and (looking-at "%")
					 (= (mod counter 2) 0))
				    (setq colmn t)
				    (setq counter 0) )
				  ( t t ))
			    (backward-char))
			  (not (and colmn
				    (= (mod counter 2) 0)))
			  (> counter 0)
			  (= (mod counter 2) 1))
			;; close string before fill point,
			;; open string anew after indenting
			(if (not eclipse-autoindent-selected)
			    (save-excursion
			      (goto-char fill-point)
			      (insert "\n"))
			  (save-excursion
			    (goto-char fill-point)
			    (cond ( (save-excursion
				      (backward-char)
				      (looking-at "\""))
				    (backward-char 2)
				    (insert "\n") )
				  ( t
				    (insert "\"\n\"") )))
			  (eclipse-indent-line)
			  (skip-chars-forward " \t")))
		      ( (save-excursion
			  ;; inside a comment?
			  (beginning-of-line)
			  (setq colmn nil)
			  (while (and (< (point) fill-point) (not colmn) )
			    (cond ( (looking-at "\"")
				    (forward-char)
				    (re-search-forward "[^\\]\"") )
				  ( (looking-at "[0-9]'[0-9a-zA-Z]+")
				    (forward-char 2) )
				  ( (looking-at "'")
				    (forward-char)
				    (re-search-forward "[^\\]'") )
				  ( (looking-at "%")
				    (setq colmn (point)) )
				  ( t (forward-char) )))
			  colmn)
 			;; continue comment in next line
			(if (not eclipse-autoindent-selected)
			    (save-excursion
			      (goto-char fill-point)
			      (insert "\n% "))
			  (save-excursion
			    (goto-char colmn)
			    (setq colmn (current-column))
			    (setq counter 0)
			    (while (looking-at "%")
			      (forward-char)
			      (setq counter (+ counter 1)))
			    (goto-char fill-point)
			    (insert "\n")
			    (indent-to colmn)
			    (while (> counter 0)
			      (insert "%")
			      (setq counter (- counter 1))))) )
		      ( t
			(save-excursion
			  (goto-char fill-point)
			  (insert "\n")
			  (if eclipse-autoindent-selected
			      (eclipse-indent-line)
			    t)
			  (setq ep (point))
			  )))
		;; If making the new line didn't reduce the hpos of
		;; the end of the line, then give up now.
		(if (>= (current-column) prev-column)
		    (setq give-up t)))
	    ;; No good place to break => stop trying.
	    (setq give-up t))))
      (if (and ep (< (point) ep))
	  (goto-char ep))
      t))) 

;;
;; Eclipse mode commenting in & out
;;

(defun eclipse-recenter ()
  "(Re-)Fontify the current buffer as ECLiPSe code and re-center it."
  (interactive)
  (let ((pos (- (point-max) (point))))
    (font-lock-fontify-buffer)
    (recenter)
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    t))

(defun eclipse-comment-region ()
  "Comment out current region."
  (interactive)
  (let ((pos (point)) (lines 0)
	(regionbegin (save-excursion (goto-char (region-beginning))
				     (beginning-of-line)
				     (point))))
    (message "Commenting out region...")
    (goto-char (region-end))
    (if (bolp) (forward-line -1))
    (beginning-of-line)
    (while (>= (point) regionbegin)
      (insert "%% ")
      (setq lines (+ lines 1))
      (forward-line -1)
      (beginning-of-line))
    (message "Commenting out region...done")
    (goto-char (+ pos (* lines 3)))
    t
    ))

(defun eclipse-uncomment-region ()
  "Uncomment current region."
  (interactive)
  (let ((pos (point)) (regionend (region-end)) (lines 0) (max 0))
    (message "Un-commenting region...")
    (goto-char (region-beginning))
    (while (< (point) regionend)
      (forward-line)
      (setq lines (+ lines 1)))
    (while (> lines 0)
      (forward-line -1)
      (setq lines (- lines 1))
      (beginning-of-line)
      (cond ( (looking-at "%% ") (delete-char 3) (setq max (+ max 3)) )
	    ( (looking-at "%%\t") (delete-char 2) (setq max (+ max 2)) )
	    ( (looking-at "%[ \t]") (delete-char 1) (setq max (+ max 1)) )
	    ( t t ) ))
    (message "Un-commenting region...done")
    (goto-char (- pos max))
    t
    ))

(defun eclipse-invert-comment-region ()
  "Invert commenting of current region."
  (interactive)
  (let ((pos (point)) (regionend (region-end)) (lines 0) (max 0))
    (message "Inverting commenting of region...")
    (goto-char (region-beginning))
    (while (< (point) regionend)
      (forward-line)
      (setq lines (+ lines 1)))
    (while (> lines 0)
      (forward-line -1)
      (setq lines (- lines 1))
      (beginning-of-line)
      (cond ( (looking-at "%% ") (delete-char 3) (setq max (+ max 3)) )
	    ( (looking-at "%%\t") (delete-char 2) (setq max (+ max 2)) )
	    ( (looking-at "%[ \t]") (delete-char 1) (setq max (+ max 1)) )
	    ( t (insert "%% ") (setq max (- max 3))) ))
    (message "Inverting commenting of region...done")
    (goto-char (- pos max))
    t
    ))

;;
;; Eclipse mode indentation
;;

(defun eclipse-tab-mode-toggle ()
  "Toggle tab-mode on/off in ECLiPSe mode."
  (interactive)
  (cond ( eclipse-tab-mode
	  (setq indent-tabs-mode nil)
	  (setq eclipse-tab-mode nil) )
	( t
	  (setq indent-tabs-mode t)
	  (setq eclipse-tab-mode t) ))
  t)

(defun eclipse-insert-tab ()
  "Insert a tab character, or, if eclipse-tab-mode is off, `eclipse-indent-width' many space characters."
  (interactive)
  (if eclipse-tab-mode
      (insert "\t")
    (let ((c 0))
      (while (< c eclipse-indent-width)
	(insert " ")
	(setq c (+ c 1))))))

(defun eclipse-indent-line (&optional af)
  "Indent current line as ECLiPSe code."
  (interactive)
  (let* ((quotes (eclipse-count-quotes))
         (indent (eclipse-indent-level quotes))
	 (pos (- (point-max) (point))) beg)	
    ;; if inside string and auto-line-break is on
    (cond ( (and af eclipse-autolinebreak-selected quotes)
	    (save-excursion
	      (forward-line -1)
	      (end-of-line)
	      (insert "\""))
	    (setq indent (+ quotes 1)) )
	  ( t t ))
    (beginning-of-line)
    (setq beg (point))
    (skip-chars-forward " \t")
    (if (zerop (- indent (current-column)))
	nil
      (delete-region beg (point))
      (indent-to indent))
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    (if (and af eclipse-autolinebreak-selected quotes)
	(insert "\"")
      t)
    t))

(defun eclipse-indent-region ()
  "Indent current region as ECLiPSe code."
  (interactive)
  (let ((pos (- (point-max) (point))))
    (message "Indenting region...")
    (indent-region (region-beginning) (region-end) nil)
    (message "Indenting region...done")
    (if (> (- (point-max) pos) (point))
	(goto-char (- (point-max) pos)))
    t))

(defun eclipse-indent-level (&optional qc)
  "Compute indentation level for next line in ECLiPSe code."
  (let ((qf nil) (cf nil))
    (save-excursion
      (beginning-of-line)
      (skip-chars-forward " \t")
      (cond ( (looking-at "\"") (setq qf t) )
	    ( (looking-at "%")
	      (save-excursion
		(forward-line -1)
		(eclipse-end-of-clause)
		(skip-chars-forward " \t")
		(cond ( (looking-at "%") (setq cf (current-column)) )
		      ( t t ))) )
	    ( (looking-at "</?[A-Z]+>") ;; html-comment
	      (setq cf (current-column)) )
	    ( t t ))
      (cond
       ;; this line a comment and we know how to indent?
       ( cf cf )
       ;; else this line a major comment?
       ( (looking-at "%%") 0 )
       ;; or a c-style comment?
       ( (looking-at "/\\*") (current-column) )
       ;; else if beginning of buffer: no indentation
       ( (bobp) 0 )
       ;; else compute indent level
       ( t (let ((empty t) ind (cl nil) (cmnt nil) pos)
	     ;; See previous indentation
	     (save-excursion
	       (forward-line -1)
	       (beginning-of-line)
	       (cond ( (and eclipse-emacs-21
			    (string-equal (buffer-name) "*eclipse*"))
		       ;; for inferior process under emacs 21
		       ;; currently not used
		       (end-of-line)
		       (skip-chars-forward " \t") )
		     ;; inferior process: if first line after prompt,
		     ;; get column where entry starts
		     ( (looking-at "[[]eclipse [1-9][0-9]*[]]:")
		       (search-forward ":")
		       (forward-char)
		       (skip-chars-forward " \t") )
		     ( t t ) ))
	     ;; jump over empty lines and multi-line comments
	     (while empty
	       (forward-line -1)
	       (beginning-of-line)
	       (if (bobp) (setq empty nil)
		 (skip-chars-forward " \t")
		 (cond ( (looking-at "[^/\n]*\\(/[^*\n]*\\)*\\*/\\([ \t]\\|/\\*.*\\*\\/\\)*\\(%.*\\)?\n")
			 (re-search-backward "/\\*")
			 (beginning-of-line)
			 (skip-chars-forward " \t") )
		       ( (not (looking-at "\\(%\\|\n\\)"))
			 (setq empty nil) )
		       ( t t ) )))
	     ;; comment?
	     (if (looking-at "%[^%]") (setq cl t))
	     (if (looking-at ":- comment") (setq cmnt t))
	     ;; set indentation width
	     (cond ( (looking-at ":-")
		     (save-excursion
		       (forward-char 2)
		       (skip-chars-forward " \t\n")
		       (setq ind (current-column)) ))
		   ( (looking-at "-->")
		     (save-excursion
		       (forward-char 3)
		       (skip-chars-forward " \t\n")
		       (setq ind (current-column)) ))
		   ( t
		     (setq ind (current-column)) )) ; beginning of clause
	     ;; See its tail
	     (eclipse-end-of-clause)
	     (or (bobp) (backward-char))
	     (if (looking-at "\\.") (setq cmnt nil))
	     (cond
	      ( (bobp) 0 )
	      ( cmnt eclipse-tab-width )
	      ;; comment: same indentation
	      ( cl (max ind 0) )
	      ;; "->", "-?->", or "#=>"
	      ( (and (looking-at ">")
		     (save-excursion (backward-char 2)
				     (looking-at "\\([^-]->\\|#=>\\)")))
		(cond
		 ;; "-?->": tab-width
		 ( (save-excursion
		     (backward-char 3)
		     (looking-at "-\\?->"))
		   eclipse-tab-width )
		 ;; else-clause: like last indent
		 ( t
		   (let ((addition eclipse-indent-width))
		     (if (save-excursion (beginning-of-line)
					 (looking-at "[ \t]*->"))
			 (setq addition 0))
		     (eclipse-goto-str-or-open-bracket "(" ")" ";")
		     (eclipse-skip-comments-and-empty-lines)
		     (+ (current-column) addition) ))))
	      ;; "(" or "[": increase indentation
	      ( (looking-at "[({[]")
		(+ (current-column) eclipse-indent-width) )
	      ;; "do": increase indentation
	      ( (and (looking-at "o")
		     (save-excursion
		       (backward-char 2)
		       (looking-at "[ \t]do")))
		(let ((addition eclipse-indent-width))
		  (if (save-excursion (beginning-of-line)
				      (looking-at "[ \t]*do"))
		      (setq addition 0))
		  (eclipse-goto-str-or-open-bracket "[({[]" "[]})]"
						    "\\([:?]-\\|-->\\)")
		  (eclipse-skip-comments-and-empty-lines)
		  (+ (current-column) addition)) )
	      ;; semicolon: indentation as line before
	      ( (looking-at ";")
		(eclipse-goto-str-or-open-bracket "[({[]" "[]})]"
						  "\\([:?]-\\|-->\\)" "[^-]->")
		(if (save-excursion
		      (backward-char 3)
		      (looking-at "\\(.[?:]-\\|-->\\)"))
		    eclipse-tab-width
		  (eclipse-skip-comments-and-empty-lines)
		  (current-column)) )
	      ;; next line
	      ( (looking-at ",")
		(eclipse-goto-str-or-open-bracket "[({[]" "[]})]"
						  "\\([:?]-\\|-->\\|;\\|[ \t]do[ \t\n]\\)"
						  "[^-]->"
						  "\\[eclipse [1-9][0-9]*\\]:")
		(eclipse-skip-comments-and-empty-lines)
		(if (looking-at "\\(:-\\|-->\\)")
		    ind
		  (current-column)) )
	      ;; end of list head: find beginning of list
	      ( (looking-at "|")
		(eclipse-goto-str-or-open-bracket "\\[" "\\]")
		(eclipse-skip-comments-and-empty-lines)
		(current-column) )
	      ;; new clause: TAB
	      ( (or (save-excursion
		      (backward-char)
		      (looking-at "[:?]-"))
		    (save-excursion
		      (backward-char 2)
		      (looking-at "-->")))
		eclipse-tab-width )
	      ;; operator
	      ( (and (looking-at "[=\\/><:]")
		     (save-excursion
		       (re-search-backward "[^=#@$`><~:\\/]")
		       (forward-char)
		       (looking-at "\\(#\\(/\\\\\\|<=\\|=<\\|>=\\|\\\\[/=]\\|[#<=>]\\)\\|\\$\\(<=\\|=<\\|[>\\]=\\|[<=>]\\)\\|::\\|<=\\|=\\(:=\\|\\\\=\\|[<=]\\)\\|>=\\|@\\(<=\\|=<\\|[>\\]=\\|[=>]\\)\\|\\\\==?\\|`\\(::\\|<[=>]\\|=<\\|[>\\]=\\|[<=>]\\)\\|~=\\|[<=>]\\)")))
		(eclipse-goto-bol-or-open-bracket)
		(eclipse-skip-comments-and-empty-lines)
		(current-column) )
              ( (looking-at "\\\\") 0 )
	      ;; expression: find beginning
	      ( (looking-at "[+*/-]")
		(setq pos (point))
		(if (eclipse-goto-last-operator)
		    t
		  (goto-char pos)
		  (eclipse-goto-bol-or-open-bracket))
		(eclipse-skip-comments-and-empty-lines)
		(current-column) )
	      ;; . : no indentation
	      ( (looking-at "[.]") 
		(if (not (save-excursion
			   (backward-char)
			   (looking-at "[.][.]")))
		    0
		  (eclipse-goto-str-or-open-bracket "\\[" "\\]")
		  (eclipse-goto-bol-or-open-bracket)
		  (eclipse-skip-comments-and-empty-lines)
		  (current-column)) )
	      ;; quote
	      ( (and qf (looking-at "\""))
		(forward-char)
		(eclipse-goto-bol-or-open-bracket)
		(eclipse-skip-comments-and-empty-lines)
		(current-column) )
	      ;; else: as last indentation
	      ( t
                (if (not qc)
		    (forward-char)
                  (beginning-of-line)
                  (forward-char qc))
		(eclipse-goto-str-or-open-bracket "[({[]" "[]})]"
						  "\\([:?]-\\|-->\\)")
		(backward-char)
		(if (not (looking-at "-"))
		    t
		  (forward-char)
		  (eclipse-skip-comments-and-empty-lines))
		(current-column) ))
	     )))
      )))

(defun eclipse-skip-comments-and-empty-lines ()
  ;; skip forward through whitespace & comments
  (let ((found nil))
    (while (not found)
      (skip-chars-forward " \t")
      (cond ( (looking-at "%")     ;; comment: skip to next line
	      (forward-line)
	      (beginning-of-line) )
	    ( (looking-at "/\\*")  ;; c-style comment: skip to end
	      (cond ( (looking-at "[^\n]*\\*/")
		      (forward-char 2)
		      (re-search-forward "\\*/")
		      (forward-char 2) )
		    ( t (setq found t) )) )
	    ( (looking-at "\n")    ;; end-of-line: skip to next line
	      (forward-line)
	      (beginning-of-line) )
	    ( t                    ;; non-empty character found
	      (setq found t) ))
      )))

(defun eclipse-count-quotes ()
  ;; count the number of double quotes in the line
  (let ((quotes nil) (editpoint (- (point) 1)) (sq nil) (dq nil) (cmt nil))
    (save-excursion
      (forward-line -1)
      (beginning-of-line)
      (while (and (not (>= (point) editpoint)) (not cmt))
	(cond ( (and (looking-at "[^\\]\"") (not sq))
		(if quotes
		    (setq quotes nil)
		  (setq quotes (current-column)))
		(setq dq (not dq)) ) 
	      ( (looking-at "[0-9]'[0-9a-zA-Z]+")
		(forward-char) )
	      ( (and (looking-at "[^\\]'") (not dq))
		(forward-char)
		(setq sq (not sq)) )
	      ( (and (looking-at "%") (not dq) (not sq))
		(setq cmt t) )
	      ( (and (looking-at "/\\*") (not dq) (not sq))
		(forward-char 2)
		(if (looking-at "[^\n]*\\*/")
		    (re-search-forward "\\*/")
		  (setq cmt t)) )
	      ( t t ))
	(forward-char))
      )
    quotes
    ))

(defun eclipse-search-backward (str)
  ;; find the next preceding string that is not preceded by an '\'
  (let ((found nil))
    (while (not found)
      (search-backward str)
      (if (save-excursion (backward-char) (looking-at "[^\\]"))
          (setq found t)))
    ))

(defun eclipse-goto-bol-or-open-bracket ()
  ;; goto beginning of line or open bracket, whatever comes first
  ;; skip over c-style comments, do not count beginning of line if line
  ;; only contains (beginning of) a c-style-comment
  (let ((found nil) (bc 0) (empty nil) (nf nil))
    (while (not (or found (and (bolp) (not empty))))
      (backward-char)
      (cond ( (looking-at "[({[]")
	      (if (= bc 0)
		  (setq found t)
		(setq bc (- bc 1)))
	      (setq empty nil) )
	    ( (looking-at "[])}]")
	      (setq bc (+ bc 1))
	      (setq empty nil) )
	    ( (and (looking-at "'")
		   (not (save-excursion 
			  (backward-char)
			  (or (looking-at "\\\\")
			      (looking-at "[0-9]'[0-9a-zA-Z]+")))))
	      (eclipse-goto-last-quote)
	      (setq empty nil) )
	    ( (looking-at "\"")
	      (eclipse-goto-last-double-quote)
	      (setq empty nil) )
	    ( (and (looking-at "/")
		   (save-excursion (backward-char)
				   (looking-at "\\*/")))
	      (eclipse-skip-c-comment)
	      (setq empty t) )
	    ( (looking-at "[ \t]")
	      t )
	    ( t
	      (setq empty nil) ))
      (save-excursion
	(forward-line -1)
	(eclipse-end-of-clause)
	(backward-char)
	(cond ( (looking-at "\\(+\\|\\*\\|/\\)")
		(setq found nil)
		(setq empty nil)
		(setq nf t) )
	      ( (and (looking-at "-")
		     (save-excursion (backward-char)
				     (not (looking-at "[?:]"))))
      		(setq found nil)
		(setq empty nil)
		(setq nf t) )
	      ( t t )))
      (if (not nf)
	  t
	(forward-line -1)
	(eclipse-end-of-clause)
	(setq nf nil)))
    (if (bolp)
	t
      (forward-char))
    ))

(defun eclipse-goto-str-or-open-bracket (bropen brclose 
					 &optional str1 str2 str3)
  ;; goto string or open bracket of type bropen, whatever comes first
  ;; skip over c-style comments
  ;; str1 is for strings that do not require special handling
  ;; if looking for "->", it *must* be passed as str2!
  ;; if looking for "[eclipse N]:", it *must* be str3!
  (let ((found nil) (rb 0))
    (while (and (not found) (not (bobp)))
      (cond ( (bolp)
	      (forward-line -1)
	      (eclipse-end-of-clause) )
	    ( t
	      (backward-char)
	      (cond ( (and str1 (looking-at str1) (= rb 0))
		      (setq found t)
		      (let* ((mbeg (match-beginning 0)) (mend (match-end 0))
			     (strl (- (- mend mbeg) 1)))
			(forward-char strl)) )
		    ( (and str2 (looking-at "[^-]->") (= rb 0)
			   ;; make sure we're not inside a suspension list
			   (save-excursion
			     (forward-char)
			     (eclipse-goto-str-or-open-bracket "[({[]" "[]})]"
							       "\\([:?]-\\|-->\\)")
			     (not (looking-at "\\["))))
		      (setq found t)
		      (forward-char 2) )
		    ( (looking-at bropen)
		      (if (= rb 0)
			  (setq found t)
			(setq rb (- rb 1))) )
		    ( (looking-at brclose)
		      (setq rb (+ rb 1)) )
		    ( (and (looking-at "'")
			   (not (save-excursion 
				  (backward-char)
				  (or (looking-at "\\\\")
				      (looking-at "[0-9]'[0-9a-zA-Z]+")))))
		      (eclipse-goto-last-quote) )
		    ( (looking-at "\"")
		      (eclipse-goto-last-double-quote) )
		    ( (and (looking-at "/")
			   (save-excursion (backward-char)
					   (looking-at "\\*/")))
		      (eclipse-skip-c-comment) )
		    ( t t ))
	      (cond ( (and str3 (looking-at str3) (= rb 0))
		      (setq found t)
		      (search-forward ":")
		      (forward-char)
		      (skip-chars-forward " \t\n")
		      (backward-char) )
		    ( t t ))) ))
    (if (bobp)
	t
      (forward-char))
    ))

(defun eclipse-goto-last-operator ()
  ;; go to the last operator
  (let ((found nil) (br 0) (comma nil))
    (while (and (not found) (not (bobp)))
      (cond ( (bolp)
	      (forward-line -1)
	      (eclipse-end-of-clause) )
	    ( t
	      (backward-char)
	      (cond ( (looking-at "[({[]")
		      (setq br (- br 1)) )
		    ( (looking-at "[])}]")
		      (setq br (+ br 1)) )
		    ( (and (looking-at "'")
			   (not (save-excursion 
				  (backward-char)
				  (or (looking-at "\\\\")
				      (looking-at "[0-9]'[0-9a-zA-Z]+")))))
		      (eclipse-goto-last-quote) )
		    ( (looking-at "\"")
		      (eclipse-goto-last-double-quote) )
		    ( (and (looking-at "/")
			   (save-excursion (backward-char)
					   (looking-at "\\*/")))
		      (eclipse-skip-c-comment) )
		    ( (and (looking-at "[=#$<>:@~`\\]")
			   (= br 0))
		      (setq found t)
		      (search-forward " ")
		      (backward-char) )
		    ( (and (looking-at "[ \t]\\(is\\|with\\)[ \t]")
			   (= br 0))
		      (setq found t)
		      (forward-char)
		      (search-forward " ")
		      (backward-char) )
		    ( (looking-at ",")
		      (setq comma t)
		      (setq found t) )
		    ( t t ) )
	      (if (> br -1)
		  t
		(setq found t)
		(setq comma t))) ))
    (if (or comma (bobp))
	nil
      (forward-char)
      t)
    ))

(defun eclipse-goto-last-quote ()
  ;; goto last quote
  (let ((found nil))
    (while (and (not found) (not (bobp)))
      (search-backward "'")
      (save-excursion (backward-char)
		      (if (or (looking-at "\\\\")
			      (looking-at "[0-9]'[0-9a-zA-Z]+"))  
			  t
			(setq found t)))
      (if (and (not found) (not (bobp)))
	  (backward-char)))))

(defun eclipse-goto-last-double-quote ()
  ;; goto last double quote
  (let ((found nil))
    (while (and (not found) (not (bobp)))
      (search-backward "\"")
      (save-excursion (backward-char)
		      (if (looking-at "\\\\")
			  t
			(setq found t)))
      (if (and (not found) (not (bobp)))
	  (backward-char)))))

(defun eclipse-skip-c-comment ()
  ;; skip backwards over c-style comments
  (backward-char)  ; we start at the last asterisk
  (re-search-backward "/\\*"))


(defun eclipse-end-of-clause ()
  ;; go to end of clause in this line
  (let* ((eolpos (save-excursion
		   (end-of-line)
		   (point)))
	 (comment nil) (flag nil) (empty t))
    (beginning-of-line)
    (while (and (not (= (point) eolpos)) (not comment))
      (cond ( (looking-at "[0-9]+'[0-9a-zA-Z]+")
	      (re-search-forward "[^0-9a-zA-Z']") )
 	    ( (looking-at "\"")
  	      (forward-char)
  	      (if (looking-at "\\\\?[ \t]*\n") (setq comment t)
  		(while (not (or (looking-at "\"") (= (point) eolpos)))
  		  (forward-char))
  		(while (not (or flag comment))
  		  (cond ( (save-excursion
			    (backward-char 2)
			    (looking-at "\\\\\""))
  			  (forward-char)
  			  (while (not (or (looking-at "\"")
					  (= (point) eolpos)))
  			    (forward-char)))
  			( (looking-at "[ \t]*\n") (setq comment t) )
  			( t (setq flag t) (forward-char) ))))
	      (setq flag nil) )
 	    ( (looking-at "'")
 	      (forward-char)
 	      (while (not (or (looking-at "'") (= (point) eolpos)))
 		(forward-char))
  	      (while (not (or flag comment))
  		(cond ( (save-excursion (backward-char) (looking-at "\\\\'"))
  			(forward-char)
  			(while (not (or (looking-at "'") (= (point) eolpos)))
  			  (forward-char)))
  		      ( (looking-at "[ \t]*\n") (setq comment t) )
  		      ( t (setq flag t) (forward-char) )))
	      (setq flag nil) )
	    ( (looking-at "%") (setq comment t) )
	    ( (looking-at "/\\*[^\n]*\\*/[ \t]*\n")
	      (setq comment t) )
	    ( (looking-at "/\\*[^\n]*\\*/")
	      (re-search-forward "\\*/")
	      (forward-char 2) )
	    ( (looking-at "\\*/[ \t]*\n")
	      ;; end of a multi-line comment: find beginning
	      (re-search-backward "/\\*")
	      (cond ( (save-excursion (beginning-of-line)
				      (looking-at "[ \t]*/\\*"))
		      (while empty
			(forward-line -1)
			(beginning-of-line)
			(if (bobp) (setq empty nil)
			  (skip-chars-forward " \t")
			  (if (not (looking-at "\\(%%\\|\n\\)"))
			      (setq empty nil)) ))
		      (setq empty t)
		      (setq eolpos (save-excursion (end-of-line) (point))) )
		    ( t (setq comment t) )))
	    ( (looking-at "/\\*") (setq comment t) )
	    ( t (forward-char) ) ))
    (skip-chars-backward " \t")
    t
    ))


;;
;; Inferior eclipse mode
;;

(defvar inferior-eclipse-mode-map nil)

(defun inferior-eclipse-mode ()
  "Major mode for interacting with an inferior ECLiPSe process.

The following commands are available:

\\{inferior-eclipse-mode-map}

Entry to this mode calls the value of `eclipse-mode-hook' with no arguments, if that value is non-nil.  Likewise with the value of `comint-mode-hook'.
`eclipse-mode-hook' is called after `comint-mode-hook'.

Commands:
Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.
\\[comint-kill-input] and \\[backward-kill-word] are kill commands, imitating normal Unix input editing.
\\[comint-interrupt-subjob] interrupts the shell or its current subjob if any.
\\[comint-stop-subjob] stops. \\[comint-quit-subjob] sends quit signal.

You can send text to the inferior ECLiPSe from other buffers using the commands `process-send-region', `process-send-string' and \\[eclipse-consult-region].

If there is a problem with entering commands in the inferior ECLiPSe process window, disable the line
     (define-key map \"\\r\" 'eclipse-next-line)
in the definition of function `eclipse-mode-commands' in the ECLiPSe mode file eclipse.el."
  (interactive)
  (require 'comint)
  (comint-mode)
  (setq major-mode 'inferior-eclipse-mode
	mode-name "Inferior Eclipse"
	comint-prompt-regexp "\\[eclipse [1-9][0-9]*\\]: ")
  (eclipse-mode-variables)
  (if inferior-eclipse-mode-map nil
    (setq inferior-eclipse-mode-map (copy-keymap comint-mode-map))
    (eclipse-mode-commands inferior-eclipse-mode-map))
  (easy-menu-define
   inferior-eclipse-process-menu inferior-eclipse-mode-map
   "ECLiPSe menu for inferior ECLiPSe process"
   '("ECLiPSe"
     ["Run ECLiPSe" run-eclipse t]
     ["Stop ECLiPSe" stop-eclipse t]
     ["Kill ECLiPSe" kill-eclipse t]
     "--"
     ["Start TkTools" eclipse-start-tools t]) )
  (easy-menu-add inferior-eclipse-process-menu)
  (use-local-map inferior-eclipse-mode-map)
  (run-hooks 'eclipse-mode-hook))

;;;###autoload
(defun run-eclipse ()
  "Run an inferior ECLiPSe process, input and output via buffer *eclipse*."
  (interactive)
  (require 'comint)
  (let (eclipse-window)
    (save-excursion
      (setq eclipse-window (get-buffer-window "*eclipse*"))
      (cond ( (not eclipse-window)
	      (split-window)
	      (select-window (next-window)) )
	    ( t (select-window eclipse-window) ))
      (switch-to-buffer (make-comint "eclipse" eclipse-program-call))
      (inferior-eclipse-mode)
      )))

(defun stop-eclipse ()
  "Send C-c to an inferior ECLiPSe process."
  (interactive)
  (let (eclipse-window eclipse-status)
    (save-excursion
      (setq eclipse-window (get-buffer-window "*eclipse*"))
      (setq eclipse-status (process-status "eclipse"))
      (cond ( (not eclipse-status)
	      (beep)
	      (message "No ECLiPSe process running") )
	    ( t
	      (process-send-string "eclipse" "\C-c")
	      (if (not eclipse-window)
		  (switch-to-buffer "*eclipse*")
		(select-window eclipse-window)) ))
      t)))

(defun kill-eclipse ()
  "Kill an inferior ECLiPSe process."
  (interactive)
  (let (eclipse-window eclipse-status exists)
    (save-excursion
      (setq eclipse-window (get-buffer-window "*eclipse*"))
      (setq eclipse-status (process-status "eclipse"))
      (if (not eclipse-status) 
	  t
	(process-send-string "eclipse" eclipse-halt-string))
      (setq exists (get-buffer "*eclipse*"))
      (cond ( (bufferp exists) (kill-buffer "*eclipse*") ) 
	    ( t t ) )
      (setq exists (get-buffer "*tktools*"))
      (cond ( (bufferp exists) (kill-buffer "*tktools*") ) 
	    ( t t ) )
      (if (not eclipse-window)
	  t
	(delete-window eclipse-window) )
      t)))

(defun eclipse-start-tools ()
  "Start TkTools for an inferior ECLiPSe process in an external Tcl/Tk window.
The socket number shown in the ECLiPSe process buffer has to be entered manually into the input field \"Port\" in the external TkTools window."
  (interactive)
  (let (eclipse-window eclipse-status current-filter eclipse-process)
    (save-excursion
      (setq eclipse-status (process-status "eclipse") current-filter)
      (cond ( (not eclipse-status)
	      (beep)
	      (message "No ECLiPSe process running") )
	    ;; if version < 5.4 there's no automatic start-up
	    ( (< eclipse-version 5.4)
	      (setq eclipse-window (get-buffer-window "*eclipse*"))
	      (if (not eclipse-window)
		  (switch-to-buffer "*eclipse*")
		(select-window eclipse-window))
	      (start-process "tktools" "*tktools*" eclipse-tktools-call)
	      (process-send-string "eclipse" eclipse-tktools-lib-call)
	      (process-send-string "eclipse" eclipse-53-tktools-call)
	      (message "Enter socket number in TkTools input field \"Port\" and press \"OK\"") )
	    ( t
	      (setq eclipse-process (get-process "eclipse"))
	      ;; add to preoutput filters
	      (add-hook 'comint-preoutput-filter-functions
			eclipse-run-tktools-func)
	      (setq eclipse-window (get-buffer-window "*eclipse*"))
	      (if (not eclipse-window)
		  (switch-to-buffer "*eclipse*")
		(select-window eclipse-window))
	      (process-send-string "eclipse" eclipse-tktools-lib-call)
	      (process-send-string "eclipse" eclipse-54-tktools-call)
	      ))
      t)))

(defun eclipse-run-tktools (output)
  ;; Extracts Host and Port parameters from output, starts TkTools
  ;; with host and port parameters.
  ;; This command needs ECLiPSe 5.4 or later.
  (let (host port output-list)
    (cond ( (string-match "\\[.*, .*\\]\n" output)
	    (setq output-list (split-string output "\\(\\[\\|\\]\\|, \\)"))
	    (setq host (nth 0 output-list))
	    (setq port (nth 1 output-list))
	    (start-process "tktools" "*tktools*" eclipse-tktools-name
			   "--" "-h" host "-p" port)
	    ;; we started tktools, so let's remove this function from the
	    ;; preoutput filters...
	    (remove-hook 'comint-preoutput-filter-functions
			 eclipse-run-tktools-func)
	    "" )
	  ( t output )
	  )))

(defun eclipse-run-region (compile beg end)
  "Send the region to an inferior ECLiPSe process."
  (interactive "P\nr")
  (let (eclipse-window)
    (save-excursion
      (process-send-region "eclipse" beg end)
      (setq eclipse-window (get-buffer-window "*eclipse*"))
      (if (not eclipse-window)
	  (switch-to-buffer "*eclipse*")
	(select-window eclipse-window)))))

(defun eclipse-compile-buffer ()
  "Send the entire buffer to an inferior ECLiPSe process and compile it."
  (interactive)
  (save-excursion
    (process-send-string "eclipse" eclipse-compile-string)
    (process-send-string "eclipse" (buffer-string))
    (process-send-string "eclipse" "\n")		;May be unnecessary
    (if eclipse-eof-string
	(process-send-string "eclipse" eclipse-eof-string)
      (process-send-eof "eclipse")) ;Send eof to eclipse process.
    t))

(defun eclipse-compile-region (compile beg end)
  "Send the region to an inferior ECLiPSe process and compile it."
  (interactive "P\nr")
  (save-excursion
    (process-send-string "eclipse" eclipse-compile-string)
    (process-send-region "eclipse" beg end)
    (process-send-string "eclipse" "\n")		;May be unnecessary
    (if eclipse-eof-string
	(process-send-string "eclipse" eclipse-eof-string)
      (process-send-eof "eclipse")) ;Send eof to eclipse process.
    t))

(defun eclipse-compile-region-and-go (compile beg end)
  "Send the region to an inferior ECLiPSe process, compile it, and switch to *eclipse* buffer."
  (interactive "P\nr")
  (eclipse-compile-region compile beg end)
  (let (eclipse-window)
    (setq eclipse-window (get-buffer-window "*eclipse*"))
    (if (not eclipse-window)
	(switch-to-buffer "*eclipse*")
      (select-window eclipse-window))
    t))

(provide 'eclipse)

;;; eclipse.el ends here
