% ----------------------------------------------------------------------
% System:	ECLiPSe Constraint Logic Programming System
% Copyright (C) Imperial College London and ICL 1995-1999
% Version:	$Id: structures.pl,v 1.8 2001/09/13 17:48:58 js10 Exp $
% ----------------------------------------------------------------------

%
% SEPIA PROLOG LIBRARY MODULE
%
% sccsid("@(#)structures.pl	30.8             95/03/30").
% sccscr("@(#)  Copyright 1990 ECRC GmbH ").
%
% IDENTIFICATION:	structures.pl
%
% AUTHOR:		Joachim Schimpf
%
% CONTENTS:
%	predicates:	define_struct(+Template)
%			erase_struct(+StructName)
%			portray_struct(+Stream, +Struct)
%
%	macros:		?Struct with ?Fields
%			+FieldName of +StructName
%
% DESCRIPTION:
%
%	NOTE: the core of this library has been moved to the kernel!
%
%	A package to allow the use of structures with field names.
%	It is intended to make programs more readable and easier
%	to modify, without compromising efficiency.
%
%	Load this library using
%
%		:- use_module(library(structures)).
%
%	A structure is declared by specifying a template like:
%
%		:- define_struct( book(author, title, year, publisher) ).
%
%	Structures with the functor book/4 can then be written as
%
%		book with []
%		book with title:'tom sawyer'
%		book with [title:'tom sawyer', year:1886, author:twain]
%
%	which translate to the corresponding forms
%
%		book(_, _, _, _)
%		book(_, 'tom sawyer', _, _)
%		book(twain, 'tom sawyer', 1886, _)
%
%	This transformation is done by macro expansion, therefore it can
%	be used in any context and is as efficient as using the structures
%	directly.
%
%	The argument index of a field in a structure can be obtained
%	using a term of the form
%
%		FieldName of StructName
%
%	E.g. to access (ie. unify) a single argument of a structure,
%	use arg/3 like this:
%
%		arg(year of book, B, Y)
%
%	which is translated into
%
%		arg(3, B, Y)
%
%	For printing structures as with/2 terms, use portray_struct/2:
%
%		portray_struct(output, B)
%
%	which will print something like
%
%		book with [author : twain, title : tom sawyer,
%						year : 1886, publisher : _g117]
%
%	Remove the structure declaration using
%
%		erase_struct(book).
%
% RESTRICTIONS:
%	The structure definitions are currently global.
%

:- module(structures).
:- export
	define_struct/1,
	erase_struct/1,
	eq_struct/2,
	portray_struct/2.

:- import
	struct_data/5
   from sepia_kernel.

:- export op(650, xfx, [eq_struct]).


:- pragma(nodebug).

:- import
	define_struct_/2,
	erase_struct_/2
   from sepia_kernel.


:- define_macro(eq_struct/2, tr_eq_struct/2, [goal]).

% the macro transformation of eq_struct/2
tr_eq_struct(no_macro_expansion(S:Sstruct eq_struct P:Pstruct),Out) :-
	atom(S),
	atom(P),
	cast_struct(S,P,Stemp,Ptemp),
	!,
	% unify the appropriate fields
	Out = (Sstruct = Stemp,Pstruct=Ptemp).
tr_eq_struct(Term, _) :-
	printf(error, "WARNING: transformation failed on %w\n", Term),
	flush(error),
	fail.


% struct_data/5 holds the structure descriptions:
%
% struct_data(FieldName, StructName, Arity, FieldIndex, Module).

% cast_/4 holds templates to cast from a struct to it's immediate parents.
%
% cast_(StructFunctor,ParentFunctor,StructTemplate,ParentTemplate).

:- dynamic cast_/4.

define_struct(Struct0:P) :-
	!,
	( atom(P) ->
		[P]=Parents
	;
		P=Parents
	),
	Struct0 =.. [Functor|StructArgs],
	concat_fields(Parents,Fields),
	append(StructArgs,Fields,AllArgs),
	Struct =.. [Functor|AllArgs],
	define_struct_(Struct),
	make_casts(Functor,Parents).
define_struct(Struct) :-
	functor(Struct, Functor, Arity),
	erase_struct(Functor),
	define_struct_(Struct).

    concat_fields([],[]).
    concat_fields([P|Ps],Fields) :-
	    findall(Field,struct_data(Field,P,_,_,_),PFields),
	    append(PFields,Fs,Fields),
	    concat_fields(Ps,Fs).
		
    make_casts(S,[]).
    make_casts(S,[P|Ps]) :-
	once(struct_data(_,S,Sarity,_,_)),
	once(struct_data(_,P,Parity,_,_)),
	findall([Spos|Ppos],
	    ( struct_data(Field,S,Sarity,Spos,_),
	      struct_data(Field,P,Parity,Ppos,_)),
	    PosPairs),
	functor(Stemplate,S,Sarity),
	functor(Ptemplate,P,Parity),
	link_pairs(PosPairs,Stemplate,Ptemplate),
	assert(cast_(S,P,Stemplate,Ptemplate)),
    	make_casts(S,Ps).

	link_pairs([],_,_).
	link_pairs([[P1|P2]|Ps],S1,S2) :-
		arg(P1,S1,X),
		arg(P2,S2,X),
		link_pairs(Ps,S1,S2).

    define_struct_(Struct) :-
	    define_struct_(Struct, structures),
	    functor(Struct, Functor, Arity),
	    assert_portray(Struct, Functor, Arity).


erase_struct(Functor) :-
	retract_all(cast_(Functor,_,_,_)),
	erase_struct_(Functor, structures),
	retract_portray(Functor).


% define appropriate rules for portray_struct/2

:- dynamic portray_struct/2.

assert_portray(Struct, Functor, Arity) :-
	functor(Template, Functor, Arity),
	make_list(Struct, Template, Arity, [], List),
	assert((
		portray_struct(Stream, Template) :-
			print(Stream, no_macro_expansion(Functor with List))
	)).

make_list(_Struct, _Template, 0, List, List) :- !.
make_list(Struct, Template, N, List0, List) :-
	arg(N, Struct, FieldName),
	arg(N, Template, FieldValue),
	N1 is N-1,
	make_list(Struct, Template, N1, [FieldName:FieldValue|List0], List).


retract_portray(Functor) :-
	clause(portray_struct(_, Template) :- _),
	functor(Template, Functor, _),
	retract((portray_struct(_, Template) :- _)),
	fail.
retract_portray(_).


eq_struct(Functor:Struct , Parent:NewStruct) :-
    cast_struct(Functor, Parent, Struct, NewStruct),
    !.

cast_struct(Functor, Parent, Struct, NewStruct) :-
	cast_(Functor, Parent, Struct, NewStruct).
cast_struct(Functor, Parent, Struct, NewStruct) :-
	cast_(Functor, Parent0, Struct, NewStruct0),
	cast_struct(Parent0, P, NewStruct0, NewStruct).

