% ----------------------------------------------------------------------
% System:	ECLiPSe Constraint Logic Programming System
% Copyright (C) Imperial College London and ICL 1995-1998
% Copyright (C) Parc Technologies Ltd 1999
% Version:	$Id: fdplex.pl,v 1.13 2000/06/20 10:05:36 js10 Exp $
%
% ECLiPSe FD/LP solver cooperation
%
% J.Schimpf, IC-Parc
% ----------------------------------------------------------------------

% ----------------------------------------------------------------------
:- module(fdplex).
% ----------------------------------------------------------------------

:- comment(summary, "An example of a hybrid solver, using lib(fd) and lib(eplex)").
:- comment(author, "Joachim Schimpf, IC-Parc").
:- comment(date, "$Date: 2000/06/20 10:05:36 $").
:- comment(copyright, "Parc Technologies").

:- use_module(library(fd)).
:- use_module(library(eplex)).
:- use_module(library(range)).

%:- pragma(nodebug).

:- export
	minimize/2,
	min_max/2,
	indomain/1,
	labeling/1,
	split_domain/1,
	split_labeling/1,
	extract_lp_from_fd/1,
	fdplex_verbosity/1,
	fd_lp_setup/3,
	transfer_bounds/1,
	fd_to_range_bounds/1,
	range_to_fd_bounds/1.

:- tool(minimize/2).
:- tool(min_max/2).


% ----------------------------------------------------------------------
% Mapping to fd and eplex
% ----------------------------------------------------------------------

:- export (::)/2, (#=)/2, (#>=)/2, (#<=)/2, (#=<)/2, (#>)/2, (#<)/2.

:- inline((::)/2, tr_constraint/2).
Vars :: Range :-
	fd:(Vars :: Range),
	range:(Vars :: Range).	% including integrality

:- inline((#=)/2, tr_constraint/2).
X #= Y :-
	fd:(X #= Y),
	eplex:(X =:= Y).		% relaxation only

:- inline((#>=)/2, tr_constraint/2).
X #>= Y :-
	fd:(X #>= Y),
	eplex:(X >= Y).	% relaxation only

:- inline((#>)/2, tr_constraint/2).
X #> Y :-
	fd:(X #> Y),
	eplex:(X >= Y+1).	% relaxation only

:- inline((#=<)/2, tr_constraint/2).
X #=< Y :-
	fd:(X #=< Y),
	eplex:(X =< Y).	% relaxation only

:- inline((#=<)/2, tr_constraint/2).
X #<= Y :-
	fd:(X #=< Y),
	eplex:(X =< Y).	% relaxation only

:- inline((#<)/2, tr_constraint/2).
X #< Y :-
	fd:(X #< Y),
	eplex:(X+1 =< Y).	% relaxation only

:- export tr_constraint/2.
tr_constraint(X :: R, (fd:(X :: R), range:(X :: R))).
tr_constraint(X #= Y, (fd:(X #= Y), eplex:(X =:= Y))).
tr_constraint(X #>= Y, (fd:(X #>= Y), eplex:(X >= Y))).
tr_constraint(X #=< Y, (fd:(X #=< Y), eplex:(X =< Y))).
tr_constraint(X #<= Y, (fd:(X #=< Y), eplex:(X =< Y))).
tr_constraint(X #> Y, (fd:(X #> Y), eplex:(X >= Y+1))).
tr_constraint(X #< Y, (fd:(X #< Y), eplex:(X+1 =< Y))).


% ----------------------------------------------------------------------
% Extract LP-constraints and variable bounds
% from the current finite-domain constraints
% ----------------------------------------------------------------------

% Transfer fd -> range bounds once (and set integer type)

transfer_bounds([]).
transfer_bounds([X|Xs]) :-
	( dvar_range(X, L, H) ->
	    range:(X :: L..H)
	;
	    report(0, "Can't translate fd domain to range: %Mw", [X])
	),
	transfer_bounds(Xs).


% Forwarding fd -> range bounds on change
% Prio 4, ie. higher than lp_solve_demon!

fd_to_range_bounds(X) :-
	var(X), !,
	range:integers([X]),
	suspend(fd_to_range_bounds(X,S), 4, [X->min,X->max], S),
	fd_to_range_bounds(X,S).
fd_to_range_bounds(X) :-
	number(X).
fd_to_range_bounds([X|Xs]) :-
	fd_to_range_bounds(X),
	fd_to_range_bounds(Xs).
fd_to_range_bounds([]).

:- demon fd_to_range_bounds/2.
fd_to_range_bounds(X,_) :-
	var(X),
	dvar_range(X, Min, Max),
	lwb(X, Min), upb(X, Max),
	wake.
fd_to_range_bounds(X,S) :-
	number(X),
	kill_suspension(S).


% Forwarding range -> fd bounds on change
% Prio 1, ie. higher than fd preds!

range_to_fd_bounds(X) :-
	var(X), !,
	var_range(X, Min, Max),
	L is fix(round(Min)),
	H is fix(round(Max)),
	fd_domain:(X :: L..H),
	suspend(range_to_fd_bounds(X,S), 1, [X->wake_lo,X->wake_hi], S).
range_to_fd_bounds(X) :-
	number(X).
range_to_fd_bounds([X|Xs]) :-
	range_to_fd_bounds(X),
	range_to_fd_bounds(Xs).
range_to_fd_bounds([]).

:- demon range_to_fd_bounds/2.
range_to_fd_bounds(X,_) :-
	var(X),
	var_range(X, Min, Max),
	MinI is fix(round(Min)),
	MaxI is fix(round(Max)),
	dvar_remove_smaller(X, MinI),
	dvar_remove_greater(X, MaxI),
	wake.
range_to_fd_bounds(X,S) :-
	number(X),
	kill_suspension(S).


% Extract constraints suitable for lp_solve
% from the current fd constraint store

extract_lp_from_fd(Cstrs) :-
	suspensions(Susps),
	collect_constraints(Susps, Cstrs).

collect_constraints([], []).
collect_constraints([Susp|Susps], Cstrs) :-
	suspension_to_goal(Susp, Goal, _Module),
	(
	    tr_fd_arith_out(Goal, Cfd),
	    fd2lp(Cfd, Clp)
	->
	    Cstrs = [Clp|Cstrs0]
	;
	    Cstrs = Cstrs0
	),
	collect_constraints(Susps, Cstrs0).

:- mode fd2lp(+,-).
fd2lp(X #=  Y,	X =:= Y).
fd2lp(X #>= Y,	X >= Y).
%fd2lp(X #=< Y,	X =< Y).
fd2lp(X #<= Y,	X =< Y).
fd2lp(X #> Y,	X >= Y+1).
fd2lp(X #< Y,	X+1 =< Y).


% ----------------------------------------------------------------------
% Labeling routines
% ----------------------------------------------------------------------

labeling([]).
labeling([X|Xs]) :-
	indomain(X),
	labeling(Xs).


indomain(X) :-
	nonvar(X).
indomain(X) :-
	var(X),
	% Make cost-variable update (in minimize_bound_check)
	% and labeling (X=...) an atomic step.
	% This is to avoid triggering propagation/re-solving twice
	call_priority(indomain_atomic(X), 2).

indomain_atomic(X) :-
	( incval(level) ; decval(level), fail ),
%	( fail ->
	( lp_var_solution(X, Val) ->
	    IVal is fix(round(Val)),
	    (
		try,
		X = IVal		% try rounded float solution first
	    ;
		dvar_domain(X, Dom),	% same as:
		dom_member(DVal, Dom),	%	X #\= IVal,
		DVal \= IVal,		%	indomain(X),
		retry,
		X = DVal
	    )
	; dvar_domain(X, Dom) ->	% same as: indomain(X)
	    decval(backtracks),
	    dom_member(Val, Dom),
	    minimize_bound_check,
	    incval(backtracks),
	    iput(0'=),
	    X = Val
	;
	    error(4, indomain(X))
	).

try :-
	iput(0'+).

retry :-
	minimize_bound_check,
	incval(backtracks),
	iput(0'-).


split_labeling([]).
split_labeling(XXs) :-
	XXs = [X|Xs],
	split_domain(X),
	( var(X) -> split_labeling(XXs) ; split_labeling(Xs) ).

split_domain(X) :-
	nonvar(X), !.
split_domain(X) :-
	( incval(level) ; decval(level), fail ),
	lp_var_solution(X, Val),
	dvar_range(X, Lo, Hi),
	( Hi-Lo > 1 -> writeln(non-binary) ; true),
	RVal is round(Val),
	IVal is fix(RVal),
	Lo =< IVal, IVal =< Hi,
	!,
	( IVal == Lo ->
	    ( try, X = IVal ; retry, X #> IVal )
	; IVal == Hi ->
	    ( try, X = IVal ; retry, X #< IVal )
	; abs(Val - RVal) < int_tolerance ->
	    ( try, X = IVal ; retry, X #> IVal ; retry, X #< IVal )
	; Val > RVal ->
	    ( try, X #<= IVal ; retry, X #> IVal )
	;
	    ( try, X #>= IVal ; retry, X #< IVal )
	).
split_domain(X) :-
%	writeln('defaulted to normal indomain'),
	dvar_domain(X, Dom),		% same as: indomain(X)
	!,
	decval(backtracks),
	dom_member(Val, Dom),
	retry,
	iput(0'=),
	X = Val.
split_domain(X) :-
	error(4, split_domain(X)).


% ----------------------------------------------------------------------
% Redefined optimization primitives
% ----------------------------------------------------------------------

:- tool(minimize/2, minimize/3).
minimize(Goal, Expr, Module) :-
	fd_lp_setup(Expr, Cost, ProblemHandle),
	report(1, "Start minimize\n", []),
	minimize_body(Goal, Cost, Module),
	print_counters(ProblemHandle, output),
	lp_cleanup(ProblemHandle).

:- tool(min_max/2, min_max/3).
min_max(Goal, Expr, Module) :-
	nonvar(Expr),
	Expr = [_|_],
	!,
	error(5, min_max(Goal, Expr), Module).
min_max(Goal, Expr, Module) :-
	fd_lp_setup(Expr, Cost, ProblemHandle),
	report(1, "Start min_max\n", []),
	min_max_body(Goal, Cost, Module),
	print_counters(ProblemHandle, output),
	lp_cleanup(ProblemHandle).


fd_lp_setup(Expr, Cost, ProblemHandle) :-
	init_counters,

	% Create a Cost variable if there is none. It is needed to
	% to transfer cost bounds between the solvers.
	Cost #= Expr,

	fd_to_range_bounds(Cost),
	range_to_fd_bounds(Cost),

	report(1, "Setting up LP demon\n", []),
	(
	    lp_demon_setup(min(Expr), Cost, [keep_basis(no)], 9,
		[
		% Different trigger modes. Find out experimentally which is
		% the best precision/speed tradeoff for your application:
		%   inst,
		    deviating_inst,
		%   bounds,
		%   deviating_bounds,
		    post(report_bounding(ProblemHandle,Cost))
		],
		ProblemHandle)
	->
	    report(1, "Initial LP problem solved\n", []),
	    report_problem(ProblemHandle)
	;
	    report(1, "LP problem infeasible\n", []),
	    fail
	).

% ----------------------------------------------------------------------
% Auxliliary/debugging/testing code
% ----------------------------------------------------------------------

:- local
	variable(backtracks),
	variable(level),
	variable(indent),
	variable(verbosity).

:-	setval(backtracks, 0),
	setval(level, 0),
	setval(indent, 0),
	setval(verbosity, 1).


rput(C) :-
	( getval(verbosity) >= 2 -> put(C) ; true ).

iput(C) :-
	( getval(verbosity) >= 2 ->
	    getval(level, Level),
	    getval(indent, Indent),
	    Diff is Level - Indent,
	    ( Diff >= 0 ->
		printf("%*c%c%b", [Diff, 0' , C])
	    ;
		printf("\n%*c%c%b", [Level, 0' , C])
	    ),
	    NewIndent is Level+1,
	    setval(indent, NewIndent)
	;
	    true
	).

init_counters :-
	setval(backtracks, 0).

print_counters(_, _) :-
	getval(verbosity, 0), !.
print_counters(ProblemHandle, Str) :-
	lp_get(ProblemHandle, statistics, [SS,SF|_]),
	getval(backtracks, F),
	write("FDPLEX: "),
	writeln(Str, (backtracks=F, solver_succs=SS, solver_fails=SF)).

:- init_counters.

fdplex_verbosity(N) :-
	setval(verbosity, N).

% print a message when verbosity is high enough
report(N, Text, Args) :-
	( N =< getval(verbosity) ->
	    write("FDPLEX: "),
	    printf(Text, Args),
	    flush(output)
	; true ).

% print some problem statistics
report_problem(ProblemHandle) :-
	lp_get(ProblemHandle,vars,VArr),
	functor(VArr,_,N),
	report(1, "LP problem has %d variables\n", [N]).

% report when the LpCost improves the lower cost bound
report_bounding(ProblemHandle, Cost) :-
	lp_get(ProblemHandle,cost,LpCost),
	var_range(Cost, Lwb, _),
	( LpCost-lp_get(feasibility_tol) > Lwb ->
	    report(1, "LP solution improves cost bound from %f to %f\n",
	    	[Lwb,LpCost])
	;
	    true
	).

% print current number of variables
report_varno(ProblemHandle) :-
	lp_get(ProblemHandle,vars,VArr),
	functor(VArr,_,N),
	count_vars(N,VArr,0,NVars),
	writeln(woken(NVars)).

    count_vars(0,_,M,M) :- !.
    count_vars(N,VArr,M0,M) :-
	arg(N,VArr,X),
	( var(X) -> +(M0,1,M1) ; M1=M0 ),
	-(N,1,N1),
	count_vars(N1,VArr,M1,M).

% ----------------------------------------------------------------------
% end_module(fdplex).
% ----------------------------------------------------------------------

